/* ====================================================================
 *
 * WinHcs
 * Copyright (c) 1997-1999. Philippe Printz.		All rights reserved.
 *
 *	The WinHcs software is OSI Certified Open Source Software.
 *	OSI Certified is a certification mark of the Open Source Initiative.
 *
 *	WinHcs maybe distributed under the terms of the	GNU General Public License.
 *
 *
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Philippe Printz
 *    for use in the HCS project."
 *
 * 4. The names "WinHcs" and "HCS" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Products derived from this software may not be called "WinHcs"
 *    nor may "WinHcs" appear in their names without prior written
 *    permission of the WinHcs.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Philippe Printz
 *    for use in the HCS project."
 *
 * THIS SOFTWARE IS PROVIDED BY PHILIPPE PRINTZ ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL PHILIPPE PRINTZ OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals and was originally based on software written by
 * Philippe C. Printz.
 *
 */

// WinHcsStatusView.cpp : implementation file
//

#include "stdafx.h"
#include "WinHcs.h"
#include "WinHcsStatusView.h"

#include "DlgIo.h"


#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusView

IMPLEMENT_DYNCREATE(CWinHcsStatusView, CFormView)

CWinHcsStatusView::CWinHcsStatusView()
	: CFormView(CWinHcsStatusView::IDD)
{
	m_netbitDisplayType = DISP_MAN1_4;
	m_x10DisplayOfs		= 0;

	m_font_w = 0;
	m_font_h = 0;

	m_varListIdx = -1;

	EnableAutomation();

	//{{AFX_DATA_INIT(CWinHcsStatusView)
	//}}AFX_DATA_INIT
}

CWinHcsStatusView::~CWinHcsStatusView()
{
}

void CWinHcsStatusView::OnFinalRelease()
{
	// When the last reference for an automation object is released
	// OnFinalRelease is called.  The base class will automatically
	// deletes the object.  Add additional cleanup required for your
	// object before calling the base class.

	CFormView::OnFinalRelease();
}

void CWinHcsStatusView::DoDataExchange(CDataExchange* pDX)
{
	CFormView::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CWinHcsStatusView)
	//}}AFX_DATA_MAP
}


BEGIN_MESSAGE_MAP(CWinHcsStatusView, CFormView)
	//{{AFX_MSG_MAP(CWinHcsStatusView)
	ON_COMMAND(ID_FILE_SAVE, OnFileSave)
	ON_COMMAND(ID_FILE_SAVE_AS, OnFileSaveAs)
	ON_BN_CLICKED(IDC_BTN_CLEAR, OnBtnClear)
	ON_BN_CLICKED(IDC_RADIO_DIO, OnRadioDio)
	ON_BN_CLICKED(IDC_RADIO_MAN1, OnRadioMan1)
	ON_BN_CLICKED(IDC_RADIO_MAN2, OnRadioMan2)
	ON_CBN_SELCHANGE(IDC_COMBO_X10, OnSelchangeComboX10)
	ON_WM_RBUTTONDOWN()
	ON_NOTIFY(NM_CLICK, IDC_VAR_LIST, OnClickVarList)
	ON_NOTIFY(NM_RCLICK, IDC_VAR_LIST, OnRclickVarList)
	ON_WM_DESTROY()
	ON_WM_SIZE()
	//}}AFX_MSG_MAP

	ON_COMMAND(ID_MENU_ON,				OnMenuOn)
	ON_COMMAND(ID_MENU_OFF,				OnMenuOff)
	ON_COMMAND(ID_MENU_TRANS,			OnMenuTransparent)
	ON_COMMAND(ID_MENU_ALL_ON,			OnMenuAllOn)
	ON_COMMAND(ID_MENU_ALL_OFF,			OnMenuAllOff)
	ON_COMMAND(ID_MENU_RMV_VAR,			OnMenuRmvVar)

END_MESSAGE_MAP()

BEGIN_DISPATCH_MAP(CWinHcsStatusView, CFormView)
	//{{AFX_DISPATCH_MAP(CWinHcsStatusView)
	DISP_PROPERTY_EX(CWinHcsStatusView, "NetbitView", AutomationGetNetbitView, AutomationSetNetbitView, VT_I2)
	DISP_PROPERTY_EX(CWinHcsStatusView, "X10View", AutomationGetX10View, AutomationSetX10View, VT_I2)
	DISP_PROPERTY_EX(CWinHcsStatusView, "Visible", AutomationGetVisible, AutomationSetVisible, VT_BOOL)
	DISP_FUNCTION(CWinHcsStatusView, "Update", AutomationUpdate, VT_EMPTY, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusView, "ClearConsole", AutomationClearConsole, VT_EMPTY, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusView, "AddVariable", AutomationAddVariable, VT_EMPTY, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusView, "RemoveVariable", AutomationRemoveVariable, VT_EMPTY, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusView, "RemoveAllVariables", AutomationRemoveAllVariables, VT_EMPTY, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusView, "Maximize", AutomationMaximize, VT_BOOL, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusView, "Minimize", AutomationMinimize, VT_BOOL, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusView, "Restore", AutomationRestore, VT_BOOL, VTS_NONE)
	//}}AFX_DISPATCH_MAP
END_DISPATCH_MAP()

// Note: we add support for IID_IWinHcsStatusView to support typesafe binding
//  from VBA.  This IID must match the GUID that is attached to the 
//  dispinterface in the .ODL file.

// {BF76971E-4DAC-11D3-B89C-0000861E056C}
static const IID IID_IWinHcsStatusView =
{ 0xbf76971e, 0x4dac, 0x11d3, { 0xb8, 0x9c, 0x0, 0x0, 0x86, 0x1e, 0x5, 0x6c } };

BEGIN_INTERFACE_MAP(CWinHcsStatusView, CFormView)
	INTERFACE_PART(CWinHcsStatusView, IID_IWinHcsStatusView, Dispatch)
END_INTERFACE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusView diagnostics

#ifdef _DEBUG
void CWinHcsStatusView::AssertValid() const
{
	CFormView::AssertValid();
}

void CWinHcsStatusView::Dump(CDumpContext& dc) const
{
	CFormView::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusView message handlers

void CWinHcsStatusView::OnDestroy() 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();
	CWnd			*pWnd  = this->GetParent();
	CListCtrl		*pList = (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	long			n, i, id;
	char			buf[80];
	CString			list;


	// Save view position
	pApp->SaveWindowPosition(pWnd, IDS_REG_SETTINGS, IDS_REG_VIEW_POS);


	// Save view variable list
	n = pList->GetItemCount();

	for (i = n - 1 ; i >= 0 ; i--) {

		pList->GetItemText(i, 0, buf, 80);

		if (strlen(buf) > 0) {

			id = (long) atol(buf);
			sprintf(buf, ":%d", id);
			list += buf;
		}
	}
	pApp->SetProfileCString(IDS_REG_SETTINGS, IDS_REG_VIEW_VAR, list);


	CFormView::OnDestroy();	
}


void CWinHcsStatusView::OnSize(UINT nType, int cx, int cy) 
{
	CFormView::OnSize(nType, cx, cy);
}

void CWinHcsStatusView::OnInitialUpdate() 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();
	CWnd			*pWnd  = this->GetParent();
	CButton			*pCBtn = (CButton*)		GetDlgItem(IDC_RADIO_MAN1);
	CComboBox		*pCBox = (CComboBox*)   GetDlgItem(IDC_COMBO_X10);
	CListCtrl		*pList = (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	CString			list;
	long			id;
	char			*sP;


	// Save view position
	pApp->RestoreWindowPosition(pWnd, IDS_REG_SETTINGS, IDS_REG_VIEW_POS);


	// Set the title
	this->SetWindowText(_T("HCSII Status View"));


	// Tooltips
	this->EnableToolTips(TRUE);


	// Retrieve initial settings
	m_netbitDisplayType = pApp->GetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_NETBIT, DISP_MAN1_4);
	m_x10DisplayOfs		= pApp->GetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_X10,	  0);

	switch (m_netbitDisplayType) {
	default:
	case DISP_MAN1_4:
		pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_MAN1);
		pCBtn->SetCheck(TRUE);

		m_netbitDisplayType = DISP_MAN1_4;
		break;
	case DISP_MAN5_8:
		pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_MAN2);
		pCBtn->SetCheck(TRUE);
		break;
	case DISP_DIO:
		pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_DIO);
		pCBtn->SetCheck(TRUE);
		break;
	}

	if (m_x10DisplayOfs < 0 || m_x10DisplayOfs > 13)
		m_x10DisplayOfs = 0;

	pCBox->SetCurSel(m_x10DisplayOfs);

	m_x10DisplayOfs *= 16;

	// Setup the variable list view
	CString str1;
	if (str1.LoadString(IDS_VAR_NAME) == 0)
		TRACE1("Error: Failed to load string: %d\n", IDS_VAR_NAME);
	else
		pList->InsertColumn(0, str1,	LVCFMT_LEFT, 150, 0);

	CString str2;
	if (str2.LoadString(IDS_VAR_VALUE) == 0)
		TRACE1("Error: Failed to load string: %d\n", IDS_VAR_VALUE);
	else
		pList->InsertColumn(1, str2,	LVCFMT_LEFT, 80, 1);

	pList->DeleteAllItems();


	list = pApp->GetProfileCString(IDS_REG_SETTINGS, IDS_REG_VIEW_VAR, "");
	id	= list.GetLength();
	sP	= list.GetBuffer(id);

	while (sP != NULL && *sP != 0) {

		if (*sP == ':') {

			sP++;
			if (sP != NULL && *sP != 0) {

				id = (long) atol(sP);

				if (id >= 0 && id < MAX_VARS)
					AddVariableDisplay((short) id);
			}
		}
		sP++;
	}

	// And update anything else always done
	OnUpdate(NULL, 0, NULL);

	// And update higher level stuff
	CFormView::OnInitialUpdate();
}



void CWinHcsStatusView::OnUpdate(CView* pSender, LPARAM lHint, CObject* pHint) 
{
	CDC					*pDC;
	CListBox			*pList;
	CListCtrl			*pListCtrl	= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	CHcsIo				*pIO		= ((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables();
	long				id, i, n, val;
	unsigned long		idx;
	char				*sP;
	char				buf[80];
	char				str[80];

	while ( (sP = ((CWinHcsStatusDoc *) GetDocument())->GetLastConsoleMsg()) != NULL) {

		// replace CR anf LF with spaces
		for (idx = 0 ; idx < strlen(sP) ; idx++) {

			if (sP[idx] == 0x0A || sP[idx] == 0x0D)
				sP[idx] = ' ';
		}

		pList = (CListBox*) GetDlgItem(IDC_CONSOLE);

		id = pList->AddString(sP);
		pList->SetTopIndex(id);
	}

	// Update all variable values
	n = pListCtrl->GetItemCount();

	for (i = 0 ; i < n ; i++) {

		pListCtrl->GetItemText(i, 0, buf, 80);

		if (strlen(buf) > 0) {

			id = (long) atol(buf);

			if (id >= 0 && id < MAX_VARS) {

				sP = pIO->GetIoLabel(id);
				sprintf(str, "%d: %s", id, sP);
				if (strcmp(buf, str) != 0)
					pListCtrl->SetItem(i, 0, LVIF_TEXT, str, 0, 0, 0, 0);

				pListCtrl->GetItemText(i, 1, buf, 80);

				if (pIO->IsBeingChanged(id) == TRUE) {

					strcpy(str, "-");
				} else {

					val = pIO->GetIdValue(id);
					sprintf(str, "%d", val);
				}

				if (strcmp(buf, str) != 0)
					pListCtrl->SetItem(i, 1, LVIF_TEXT, str, 0, 0, 0, 0);
			}
		}
	}

	
	// to redraw state boxes
	pDC = this->GetDC();

	OnPrepareDC(pDC, NULL);
	this->OnDraw(pDC);

	this->ReleaseDC(pDC);
}



#define	X_ORG_MODULES	56
#define	Y_ORG_MODULES	26
#define	X_ORG_INPUT		56
#define	Y_ORG_INPUT		165
#define	X_ORG_OUTPUT	56
#define	Y_ORG_OUTPUT	284
#define	X_ORG_NETBITS	336
#define	Y_ORG_NETBITS	90
#define	X_ORG_X10		110
#define	Y_ORG_X10		498
#define	X_ORG_ADCS		40
#define	Y_ORG_ADCS		410
#define	X_ORG_NET_ADCS	318
#define	Y_ORG_NET_ADCS	275
#define	X_ORG_NET_DACS	318
#define	Y_ORG_NET_DACS	410

#define	X_WIDTH			16
#define	Y_HEIGHT		10
#define	X_SPACE			10
#define	Y_SPACE			8

#define	X_SPACE_AN		24
#define	Y_SPACE_AN		10
#define	X_SPACE_AN1		2
#define	Y_SPACE_AN1		6


#define	TOOLTIP_BUF_SZ	80

void CWinHcsStatusView::ShowRS232State(void) 
{
	CWinHcsStatusDoc	*pDoc = (CWinHcsStatusDoc *) GetDocument();
	COLORREF			col;
	CDC					*pDC;

	pDC = this->GetDC();

	OnPrepareDC(pDC, NULL);

	// Update RS232 settings
	if (pDoc->IsConnectedToHcs()) {

		switch (pDoc->m_rs232_in_state) {
		default:
		case RS232_ERROR:
			col = 0x000000FF;
			break;
		case RS232_ACTIVE:
			col = 0x0000FF00;
			break;
		case RS232_NOT_ACTIVE:
			col = 0x00FFFFFF;
			break;
		}
	} else
		col = ::GetSysColor(CTLCOLOR_DLG);

	pDC->FillSolidRect(X_ORG_MODULES + 155,Y_ORG_MODULES + 86, 16, 10, col);


	if (pDoc->IsConnectedToHcs()) {

		switch (pDoc->m_rs232_out_state) {
		default:
		case RS232_ERROR:
			col = 0x000000FF;
			break;
		case RS232_ACTIVE:
			col = 0x0000FF00;
			break;
		case RS232_NOT_ACTIVE:
			col = 0x00FFFFFF;
			break;
		}
	} else
		col = ::GetSysColor(CTLCOLOR_DLG);

	pDC->FillSolidRect(X_ORG_MODULES + 60, Y_ORG_MODULES + 86, 16, 10, col);

	this->ReleaseDC(pDC);
}

void CWinHcsStatusView::OnDraw(CDC* pDC) 
{
	CWinHcsStatusDoc	*pDoc = (CWinHcsStatusDoc *) GetDocument();
	CHcsIo				*pIO;
	COLORREF			bk = ::GetSysColor(CTLCOLOR_DLG);
	COLORREF			col;
	RECT				rect;

	long				x, y, t, c, l, id, id_ofs, w, h, sw, sh;
	long				l_max, c_max, x_ofs, y_ofs, val;
	char				buf[32];

	BOOL				bDrawTest;


	// Get Font size if not already done
	if (m_font_w == 0 || m_font_h == 0) {

		rect.left	= 0;
		rect.right	= 0;
		rect.top	= 0;
		rect.bottom = 0;

		col = ::GetSysColor(COLOR_GRAYTEXT);

		pDC->SetTextColor(col);
		pDC->SetBkColor(bk);
		pDC->SetBkMode(TRANSPARENT);

		pDC->DrawText(_T("4095"), -1, &rect, DT_CALCRECT | DT_CENTER | DT_VCENTER | DT_NOPREFIX | DT_SINGLELINE);

		m_font_w = rect.right - rect.left + 5;
		m_font_h = rect.bottom - rect.top + 10;
	}


	// Update RS232 settings
	if (pDoc->IsConnectedToHcs()) {

		switch (pDoc->m_rs232_in_state) {
		default:
		case RS232_ERROR:
			col = 0x000000FF;
			break;
		case RS232_ACTIVE:
			col = 0x0000FF00;
			break;
		case RS232_NOT_ACTIVE:
			col = 0x00FFFFFF;
			break;
		}
	} else
		col = ::GetSysColor(CTLCOLOR_DLG);

	pDC->FillSolidRect(X_ORG_MODULES + 155,Y_ORG_MODULES + 86, 16, 10, col);


	if (pDoc->IsConnectedToHcs()) {

		switch (pDoc->m_rs232_out_state) {
		default:
		case RS232_ERROR:
			col = 0x000000FF;
			break;
		case RS232_ACTIVE:
			col = 0x0000FF00;
			break;
		case RS232_NOT_ACTIVE:
			col = 0x00FFFFFF;
			break;
		}
	} else
		col = ::GetSysColor(CTLCOLOR_DLG);

	pDC->FillSolidRect(X_ORG_MODULES + 60, Y_ORG_MODULES + 86, 16, 10, col);



	// And scan through all display objects to update settings
	for (t = 0 ; t < IO_MAX ; t++) {

		bDrawTest = FALSE;

		switch (t) {
		default:
		case IO_PL:					// modules  PL
			pIO		= pDoc->GetHcsPLs();
			l_max	= 1;
			c_max	= 1;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;
		case IO_LCD:					// modules  LCD
			pIO = pDoc->GetHcsLCDs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 18;				;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;
		case IO_MCIR:					// modules  MCIR
			pIO = pDoc->GetHcsMCIRs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 34;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;
		case IO_MAN:					// modules  MAN
			pIO = pDoc->GetHcsANSWERMANs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 52;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;
		case IO_DIO:					// modules  DIO
			pIO = pDoc->GetHcsDIOs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 70;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;

		case IO_INPUTS:					// Digital inputs
			pIO		= pDoc->GetHcsInputs();
			l_max	= 5;
			c_max	= 8;
			x_ofs	= X_ORG_INPUT;
			y_ofs	= Y_ORG_INPUT;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;
		case IO_OUTPUTS:					// Digital outputs
			pIO = pDoc->GetHcsOutputs();
			l_max	= 5;
			c_max	= 8;
			x_ofs	= X_ORG_OUTPUT;
			y_ofs	= Y_ORG_OUTPUT;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			break;
		case IO_NETBITS:					// Digital netbits
			pIO		= pDoc->GetHcsNetbits();
			l_max	= 8;
			c_max	= 8;
			x_ofs	= X_ORG_NETBITS;
			y_ofs	= Y_ORG_NETBITS;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;

			switch(m_netbitDisplayType) {
			default:
			case DISP_MAN1_4:
				id_ofs = 96;
				break;
			case DISP_MAN5_8:
				id_ofs = 160;
				break;
			case DISP_DIO:
				id_ofs = 0;
				break;
			}
			break;
		case IO_X10:					// X10
			pIO		= pDoc->GetHcsX10();
			l_max	= 3;
			c_max	= 16;
			x_ofs	= X_ORG_X10;
			y_ofs	= Y_ORG_X10;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= m_x10DisplayOfs;
			break;

		case IO_ADCS:					// ADCs
			pIO =  pDoc->GetHcsADCs();
			
			col = ::GetSysColor(COLOR_GRAYTEXT);

			pDC->SetTextColor(col);
			pDC->SetBkColor(bk);
			pDC->SetBkMode(TRANSPARENT);

			l_max	= 2;
			c_max	= 4;
			x_ofs	= X_ORG_ADCS;
			y_ofs	= Y_ORG_ADCS;
			w		= m_font_w;
			h		= m_font_h;
			sw		= X_SPACE_AN;
			sh		= Y_SPACE_AN;

			id_ofs	= 0;

			bDrawTest = TRUE;
			break;

		case IO_NET_ADCS:					// ADCs
			// Adcs on AMAN boards: 4 AMAn with up to 6 adcs each
			if (m_netbitDisplayType == DISP_MAN5_8)	id_ofs = 48;
			else									id_ofs = 16;

			pIO =  pDoc->GetHcsADCs();
			
			col = ::GetSysColor(COLOR_GRAYTEXT);

			pDC->SetTextColor(col);
			pDC->SetBkColor(bk);
			pDC->SetBkMode(TRANSPARENT);

			l_max	= 4;
			c_max	= 6;
			x_ofs	= X_ORG_NET_ADCS;
			y_ofs	= Y_ORG_NET_ADCS;
			w		= m_font_w;
			h		= m_font_h;
			sw		= X_SPACE_AN1;
			sh		= Y_SPACE_AN1;

			strcpy(buf, _T(""));

			bDrawTest = TRUE;
			break;

		case IO_NET_DACS:					// DACs
			pIO		= pDoc->GetHcsDACs();

			if (m_netbitDisplayType == DISP_MAN5_8)	id_ofs = 8;
			else									id_ofs = 0;

			col = ::GetSysColor(COLOR_GRAYTEXT);

			pDC->SetTextColor(col);
			pDC->SetBkColor(bk);
			pDC->SetBkMode(TRANSPARENT);

			l_max	= 2;
			c_max	= 4;
			x_ofs	= X_ORG_NET_DACS;
			y_ofs	= Y_ORG_NET_DACS;
			w		= m_font_w;
			h		= m_font_h;
			sw		= X_SPACE_AN;
			sh		= Y_SPACE_AN;

			strcpy(buf, _T(""));

			bDrawTest = TRUE;
			break;
		}



		// and draw the boxes or display the value
		for (l = 0 ; l < l_max ; l++) {

			for (c = 0 ; c < c_max ; c++) {

				if (t == IO_NET_ADCS)
					id = l * 8 + c + id_ofs;
				else if (t == IO_NET_DACS) {
					if (c < 2)
						id = l * 8 + c + id_ofs;
					else
						id = l * 8 + 2 + c + id_ofs;
				} else
					id = l * c_max + c + id_ofs;

				if (bDrawTest == FALSE) {

					if (pDoc->IsConnectedToHcs())
						col = pIO->GetColorIdState(id);
					else
						col = ::GetSysColor(CTLCOLOR_DLG);

					x = x_ofs + c * (w  + sw);
					y = y_ofs + l * (h + sh);

					pDC->FillSolidRect(x, y, w, h, col);

				} else {

					if (pDoc->IsConnectedToHcs() && pIO->IsInUse(id) && (t == IO_ADCS || m_netbitDisplayType != DISP_DIO)) {
						
						if (pIO->IsBeingChanged(id)) {

							strcpy(buf, _T("-"));

						} else  {

							val = pIO->GetIdValue(id);
							sprintf(buf, "%d", val);
						}

					} else {

						strcpy(buf, _T(""));
					}

					rect.left	= x_ofs + c * (w + sw);
					rect.right	= rect.left + w;
					rect.top	= y_ofs + l * (h + sh);
					rect.bottom = rect.top + h;

					pDC->Rectangle(&rect);
					pDC->DrawText(buf, -1, &rect, DT_CENTER | DT_VCENTER | DT_NOPREFIX | DT_SINGLELINE);
				}
			}
		}
	}
}



int CWinHcsStatusView::PointToIo(CPoint point, CHcsIo **ppIO, long *pID, long *pType, char *pszName) const
{
	CWinHcsStatusDoc	*pDoc = (CWinHcsStatusDoc *) GetDocument();
	CHcsIo				*pIO;
	long				ret = -1;
	long				s, t, c, l, id, x, id_ofs;
	long				w, h, sw, sh;
	long				l_max, c_max, x_ofs, y_ofs;
	char				*pszInfo, *pszLabel;
	CPoint				testPoint, pt;
	CListBox			*pList;
	CListCtrl			*pListCtrl;
	CRect				rect;

	
	// test point
	testPoint = this->GetScrollPosition();
	testPoint.x += point.x;
	testPoint.y += point.y;

	// scan through all possibilities
	for (t = 0 ; t < IO_MAX && ret == -1 ; t++) {

		switch (t) {
		default:
		case IO_PL:					// modules  PL
			pIO		= pDoc->GetHcsPLs();
			l_max	= 1;
			c_max	= 1;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "PL";
			break;
		case IO_LCD:					// modules  LCD
			pIO		= pDoc->GetHcsLCDs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 18;				;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "LCD";
			break;
		case IO_MCIR:					// modules  MCIR
			pIO		= pDoc->GetHcsMCIRs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 34;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "MCIR";
			break;
		case IO_MAN:					// modules  MAN
			pIO = pDoc->GetHcsANSWERMANs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 52;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "AMAN";
			break;
		case IO_DIO:					// modules  DIO
			pIO		= pDoc->GetHcsDIOs();
			l_max	= 1;
			c_max	= 8;
			x_ofs	= X_ORG_MODULES;
			y_ofs	= Y_ORG_MODULES + 70;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "DIO";
			break;

		case IO_INPUTS:					// Digital inputs
			pIO		= pDoc->GetHcsInputs();
			l_max	= 5;
			c_max	= 8;
			x_ofs	= X_ORG_INPUT;
			y_ofs	= Y_ORG_INPUT;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "Input";
			break;
		case IO_OUTPUTS:					// Digital outputs
			pIO		= pDoc->GetHcsOutputs();
			l_max	= 5;
			c_max	= 8;
			x_ofs	= X_ORG_OUTPUT;
			y_ofs	= Y_ORG_OUTPUT;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= 0;
			pszLabel= "Output";
			break;
		case IO_NETBITS:					// Digital netbits
			pIO		= pDoc->GetHcsNetbits();
			l_max	= 8;
			c_max	= 8;
			x_ofs	= X_ORG_NETBITS;
			y_ofs	= Y_ORG_NETBITS;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;

			switch(m_netbitDisplayType) {
			default:
			case DISP_MAN1_4:
				id_ofs = 96;
				pszLabel= "AMAN";
				break;
			case DISP_MAN5_8:
				id_ofs = 160;
				pszLabel= "AMAN";
				break;
			case DISP_DIO:
				id_ofs = 0;
				pszLabel= "DIO";
				break;
			}
			break;
		case IO_X10:					// X10
			pIO		= pDoc->GetHcsX10();
			l_max	= 3;
			c_max	= 16;
			x_ofs	= X_ORG_X10;
			y_ofs	= Y_ORG_X10;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;
			id_ofs	= m_x10DisplayOfs;
			pszLabel= "X10";
			break;

		case IO_ADCS:					// ADCs
			pIO =  pDoc->GetHcsADCs();

			l_max	= 2;
			c_max	= 4;
			x_ofs	= X_ORG_ADCS;
			y_ofs	= Y_ORG_ADCS;
			w		= m_font_w;
			h		= m_font_h;
			sw		= X_SPACE_AN;
			sh		= Y_SPACE_AN;

			id_ofs	= 0;

			pszLabel= "ADC";
			break;

		case IO_NET_ADCS:
			// Adcs on AMAN boards: 4 AMAn with up to 6 adcs each
			if (m_netbitDisplayType == DISP_MAN5_8)	id_ofs = 48;
			else									id_ofs = 16;

			pIO		=  pDoc->GetHcsADCs();

			l_max	= 4;
			c_max	= 6;
			x_ofs	= X_ORG_NET_ADCS;
			y_ofs	= Y_ORG_NET_ADCS;
			w		= m_font_w;
			h		= m_font_h;
			sw		= X_SPACE_AN1;
			sh		= Y_SPACE_AN1;

			if (m_netbitDisplayType == DISP_DIO) {

				l_max = 0;
				c_max = 0;
			}

			pszLabel= "ADC";
			break;

		case IO_NET_DACS:					// DACs
			pIO		= pDoc->GetHcsDACs();

			l_max	= 2;
			c_max	= 4;
			x_ofs	= X_ORG_NET_DACS;
			y_ofs	= Y_ORG_NET_DACS;
			w		= m_font_w;
			h		= m_font_h;
			sw		= X_SPACE_AN;
			sh		= Y_SPACE_AN;

			if (m_netbitDisplayType == DISP_MAN5_8)	id_ofs = 8;
			else  									id_ofs = 0;

			if (m_netbitDisplayType == DISP_DIO) {

				l_max = 0;
				c_max = 0;
			}

			pszLabel= "DAC";
			break;

		case IO_CONSOLE:				// Console
			pList	= (CListBox*) GetDlgItem(IDC_CONSOLE);
			pt		= point;

			::ClientToScreen(this->m_hWnd, &pt);
			::GetWindowRect(pList->m_hWnd, rect);

			if (rect.PtInRect(pt)) {

				*ppIO	= NULL;
				*pType	= IO_CONSOLE;
				*pID	= 0;

				strcpy(pszName, _T("Console Messages"));

				return IDC_CONSOLE;
			}

			pIO		= NULL;
			l_max	= 0;
			c_max	= 0;
			x_ofs	= 0;
			y_ofs	= 0;
			id_ofs	= 0;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;

			break;

		case IO_VARS:				// Variables
			pListCtrl= (CListCtrl*) GetDlgItem(IDC_VAR_LIST);
			pt		 = point;

			::ClientToScreen(this->m_hWnd, &pt);
			::GetWindowRect(pListCtrl->m_hWnd, rect);

			if (rect.PtInRect(pt)) {

				*ppIO	= NULL;
				*pType	= IO_CONSOLE;
				*pID	= 0;

				strcpy(pszName, _T("Variables (Left click on row to update value)"));

				return IDC_VAR_LIST;
			}

			pIO		= NULL;
			l_max	= 0;
			c_max	= 0;
			x_ofs	= 0;
			y_ofs	= 0;
			id_ofs	= 0;
			w		= X_WIDTH;
			h		= Y_HEIGHT;
			sw		= X_SPACE;
			sh		= Y_SPACE;

			break;
		}


		// And search for I/O
		for (l = 0 ; l < l_max && ret == -1 ; l++) {

			s = y_ofs + l * (h + sh);

			if (testPoint.y > s && testPoint.y < (s + h + sh)) {

				for (c = 0 ; c < c_max && ret == -1 ; c++) {

					s = x_ofs + c * (w + sw);

					if (testPoint.x > s && testPoint.x < (s + w)) {

						if (t == IO_NET_ADCS)
							id = l * 8 + c + id_ofs;
						else if (t == IO_NET_DACS) {
							if (c < 2)
								id = l * 8 + c + id_ofs;
							else
								id = l * 8 + 2 + c + id_ofs;
						} else
							id = l * c_max + c + id_ofs;

						if (pszName != NULL) {

							pszInfo = pIO->GetIoLabel(id);

							if (pszInfo != NULL) {

								if (strlen(pszInfo) > 0) {

									strncpy(pszName, pszInfo, TOOLTIP_BUF_SZ);
									pszName[TOOLTIP_BUF_SZ - 1];

								} else {

									if (t != 8)	// Non X10

										sprintf(pszName, "%s %d", pszLabel, id);

									else {

										x = 'A' + l + m_x10DisplayOfs / 16;
										sprintf(pszName, "X10 %c:%d", x, c + 1);
									}
								}

							} else {

								strcpy(pszName, _T(""));	// I/O not in use
								ret = -1;				// -> do not display help
							}
						}

						ret		= t * 20000 + id;
						*ppIO	= pIO;
						*pType	= t;
						*pID	= id;

						return ret;
					}
				}
			}
		}
	}


	*ppIO	= NULL;
	*pType	= -1;
	*pID	= -1;

	return -1;
}




int CWinHcsStatusView::OnToolHitTest(CPoint point, TOOLINFO* pTI) const
{
	CHcsIo				*pIO;
	long				ret;
	long				type, id;
	char				*pszInfo;

	pszInfo = (char *) malloc(TOOLTIP_BUF_SZ);
	ret = PointToIo(point, &pIO, &id, &type, pszInfo);

	// Return information...
	switch( ret ) {
	case IDC_CONSOLE:
		pTI->lpszText	= pszInfo;
		pTI->hinst		= NULL;
		pTI->uId		= (UINT) ((CListBox*) GetDlgItem(IDC_CONSOLE))->m_hWnd;
		pTI->hwnd		= m_hWnd;
		pTI->uFlags	   |= TTF_IDISHWND | TTF_NOTBUTTON | TTF_CENTERTIP;
		break;

	case IDC_VAR_LIST:
		pTI->lpszText	= pszInfo;
		pTI->hinst		= NULL;
		pTI->uId		= (UINT) ((CListBox*) GetDlgItem(IDC_VAR_LIST))->m_hWnd;
		pTI->hwnd		= m_hWnd;
		pTI->uFlags	   |= TTF_IDISHWND | TTF_NOTBUTTON | TTF_CENTERTIP;
		break;

	default:
		pTI->lpszText	= pszInfo;
		pTI->hinst		= NULL;
		pTI->uId		= (UINT) m_hWnd;
		pTI->hwnd		= m_hWnd;
		pTI->uFlags	   |= TTF_IDISHWND;
		break;

	case -1:
		free(pszInfo);
		break;
	}

	return ret;
}



void CWinHcsStatusView::OnFileSave() 
{
	// TODO: Add your command handler code here
	
}

void CWinHcsStatusView::OnFileSaveAs() 
{
	// TODO: Add your command handler code here
	
}



void CWinHcsStatusView::OnBtnClear() 
{
	CListBox			*pList;

	pList = (CListBox*) GetDlgItem(IDC_CONSOLE);
	pList->ResetContent();
}


void CWinHcsStatusView::OnRadioDio() 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();

	m_netbitDisplayType = DISP_DIO;	
	pApp->SetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_NETBIT, m_netbitDisplayType);

	OnUpdate(NULL, 0, NULL);
}

void CWinHcsStatusView::OnRadioMan1() 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();

	m_netbitDisplayType = DISP_MAN1_4;
	pApp->SetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_NETBIT, m_netbitDisplayType);

	OnUpdate(NULL, 0, NULL);
}

void CWinHcsStatusView::OnRadioMan2() 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();

	m_netbitDisplayType = DISP_MAN5_8;
	pApp->SetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_NETBIT, m_netbitDisplayType);

	OnUpdate(NULL, 0, NULL);
}


void CWinHcsStatusView::OnSelchangeComboX10() 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();
	CComboBox		*pCBox = (CComboBox*) GetDlgItem(IDC_COMBO_X10);

	m_x10DisplayOfs  = pCBox->GetCurSel();
	pApp->SetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_X10, m_x10DisplayOfs);
	
	m_x10DisplayOfs *= 16;

	OnUpdate(NULL, 0, NULL);
}

void CWinHcsStatusView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	CMenu			PopMenu;
	BOOL			bExists = FALSE;
	CPoint			ClientPoint = point;
	CHcsIo			*pIO;
	long			ret, flags;
	char			szTooltipBuf[TOOLTIP_BUF_SZ];

	// Popups only allowed if we are conencted to the HCS
	if (((CWinHcsStatusDoc *) GetDocument())->IsConnectedToHcs() == FALSE)
		return;

	flags = MF_STRING | MF_ENABLED;


	if ((ret = PointToIo(point, &pIO, &m_id, &m_type, szTooltipBuf)) >= 0) {

		// First remember some settings
		pIO->SetCurrentSelection(m_id);
		m_pIO = pIO;


		// And attempt to build the popup menus
		try
		{
			switch (m_type) {
			default:
			case IO_PL:					// modules  PL
			case IO_LCD:				// modules  LCD
			case IO_MCIR:				// modules  MCIR
			case IO_MAN:				// modules  MAN
			case IO_DIO:				// modules  DIO
				break;

			case IO_INPUTS:				// Digital inputs
			case IO_OUTPUTS:			// Digital outputs
			case IO_NETBITS:			// Digital netbits
				// Create popup menu
				bExists = PopMenu.CreatePopupMenu();
				if (!bExists)
					AfxThrowUserException();

				// Add popup menu entries
				if (!PopMenu.AppendMenu(flags,	ID_MENU_ON,  _T("On")) ||
					!PopMenu.AppendMenu(flags,	ID_MENU_OFF, _T("Off")))
					AfxThrowUserException();

				if (m_type == IO_INPUTS) {
					if (!PopMenu.AppendMenu(MF_SEPARATOR| MF_ENABLED,	0,	_T(""))	||
						!PopMenu.AppendMenu(flags,	ID_MENU_TRANS, _T("Transparent")))
						AfxThrowUserException();
				}
				break;

			case IO_ADCS:				// ADCs
			case IO_NET_ADCS:			// ADCs
				// Create popup menu
				bExists = PopMenu.CreatePopupMenu();
				if (!bExists)
					AfxThrowUserException();

				// Add popup menu entries
				if (!PopMenu.AppendMenu(flags,	ID_COMMANDS_SETADC, _T("Set...")))
						AfxThrowUserException();
				break;

			case IO_NET_DACS:			// DACs
				// Create popup menu
				bExists = PopMenu.CreatePopupMenu();
				if (!bExists)
					AfxThrowUserException();

				// Add popup menu entries
				if (!PopMenu.AppendMenu(flags,	ID_COMMANDS_SETDAC, _T("Set...")))
						AfxThrowUserException();
				break;

			case IO_X10:					// X10
				pIO->m_userData1 = 4;

				bExists = TRUE;

				bExists = PopMenu.CreatePopupMenu();
				if (!bExists)
					AfxThrowUserException();

				// Add popup menu entries
				if (!PopMenu.AppendMenu(flags,						ID_MENU_ON,		_T("On"))		||
					!PopMenu.AppendMenu(flags,						ID_MENU_OFF,	_T("Off"))		||
					!PopMenu.AppendMenu(flags					,	ID_COMMANDS_SETX10,_T("Brightness..."))||
					!PopMenu.AppendMenu(MF_SEPARATOR| MF_ENABLED,	0,				_T(""))			||
					!PopMenu.AppendMenu(flags,						ID_MENU_ALL_ON,  _T("All On"))	||
					!PopMenu.AppendMenu(flags,						ID_MENU_ALL_OFF, _T("All Off")))
						AfxThrowUserException();
				break;
			}


			if (bExists) {

				// Add display & track popup menus
				ClientToScreen(&ClientPoint);

				if (!PopMenu.TrackPopupMenu(TPM_LEFTALIGN, ClientPoint.x, ClientPoint.y, GetParent()))
					AfxThrowUserException();
			}

		}
		catch (CUserException* pEx)
		{
			MessageBeep(0);
			TRACE0("Warning: The popup couldn't be created!\n");

			pEx->Delete();
		}

		if (bExists)
			PopMenu.DestroyMenu();
	}
	//	CFormView::OnRButtonDown(nFlags, point);
}




void CWinHcsStatusView::OnMenuOn() 
{
	CWinHcsStatusDoc	*pDoc = (CWinHcsStatusDoc *) GetDocument();
	char				cmd[8];

	switch(m_type) {
	default:
	case IO_PL:					// modules  PL
	case IO_LCD:				// modules  LCD
	case IO_MCIR:				// modules  MCIR
	case IO_MAN:				// modules  MAN
	case IO_DIO:				// modules  DIO
	case IO_ADCS:				// ADCs
	case IO_NET_ADCS:			// ADCs
	case IO_NET_DACS:			// DACs
		break;
	case IO_INPUTS:				// Digital inputs
		pDoc->GetHcsInputs()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x15;
		cmd[2] = (char) m_id;
		cmd[3] = (char) 1;

		pDoc->SendMessageToHcs(cmd, 4);
		break;
	case IO_OUTPUTS:			// Digital outputs
		pDoc->GetHcsOutputs()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x18;
		cmd[2] = (char) m_id;
		cmd[3] = (char) 1;

		pDoc->SendMessageToHcs(cmd, 4);
		break;
	case IO_NETBITS:			// Digital netbits
		pDoc->GetHcsNetbits()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x23;
		*((unsigned short *) &cmd[2]) =  (unsigned short) m_id;
		cmd[4] = (char) 1;

		pDoc->SendMessageToHcs(cmd, 5);
		break;
	case IO_X10:					// X10
		pDoc->GetHcsX10()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x12;
		cmd[2] = (char) m_id;
		cmd[3] = (char) 2;
		cmd[4] = (char) 1;

		pDoc->SendMessageToHcs(cmd, 5);
		break;
	}
}


void CWinHcsStatusView::OnMenuOff() 
{
	CWinHcsStatusDoc	*pDoc = (CWinHcsStatusDoc *) GetDocument();
	char				cmd[8];

	switch(m_type) {
	default:
	case IO_PL:					// modules  PL
	case IO_LCD:				// modules  LCD
	case IO_MCIR:				// modules  MCIR
	case IO_MAN:				// modules  MAN
	case IO_DIO:				// modules  DIO
	case IO_ADCS:				// ADCs
	case IO_NET_ADCS:			// ADCs
	case IO_NET_DACS:			// DACs
		break;
	case IO_INPUTS:				// Digital inputs
		pDoc->GetHcsInputs()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x15;
		cmd[2] = (char) m_id;
		cmd[3] = (char) 0;

		pDoc->SendMessageToHcs(cmd, 4);
		break;
	case IO_OUTPUTS:			// Digital outputs
		pDoc->GetHcsOutputs()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x18;
		cmd[2] = (char) m_id;
		cmd[3] = (char) 0;

		pDoc->SendMessageToHcs(cmd, 4);
		break;
	case IO_NETBITS:			// Digital netbits
		pDoc->GetHcsNetbits()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x23;
		*((unsigned short *) &cmd[2]) =  (unsigned short) m_id;
		cmd[4] = (char) 0;

		pDoc->SendMessageToHcs(cmd, 5);
		break;
	case IO_X10:					// X10
		pDoc->GetHcsX10()->SetBeingChanged(m_id);
		cmd[0] = '!';
		cmd[1] = 0x12;
		cmd[2] = (char) m_id;
		cmd[3] = (char) 3;
		cmd[4] = (char) 1;

		pDoc->SendMessageToHcs(cmd, 5);
		break;
	}
}


void CWinHcsStatusView::OnMenuTransparent() 
{
	CWinHcsStatusDoc	*pDoc = (CWinHcsStatusDoc *) GetDocument();
	char				cmd[8];

	if (m_type != IO_INPUTS)
		return;

	pDoc->GetHcsInputs()->SetBeingChanged(m_id);
	cmd[0] = '!';
	cmd[1] = 0x15;
	cmd[2] = (char) m_id;
	cmd[3] = (char) 2;

	((CWinHcsStatusDoc *) GetDocument())->SendMessageToHcs(cmd, 4);
}


void CWinHcsStatusView::OnMenuAllOn() 
{
	char	cmd[8];

	if (m_type != IO_X10)
		return;

	cmd[0] = '!';
	cmd[1] = 0x12;
	cmd[2] = (char) m_id;
	cmd[3] = (char) 1;
	cmd[4] = (char) 1;

	((CWinHcsStatusDoc *) GetDocument())->SendMessageToHcs(cmd, 5);
}


void CWinHcsStatusView::OnMenuAllOff() 
{
	char	cmd[8];

	if (m_type != IO_X10)
		return;

	cmd[0] = '!';
	cmd[1] = 0x12;
	cmd[2] = (char) m_id;
	cmd[3] = (char) 0;
	cmd[4] = (char) 1;

	((CWinHcsStatusDoc *) GetDocument())->SendMessageToHcs(cmd, 5);
}


void CWinHcsStatusView::OnClickVarList(NMHDR* pNMHDR, LRESULT* pResult) 
{
	CListCtrl		*pList		= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	LV_DISPINFO		*pDispInfo	= (LV_DISPINFO*)pNMHDR;
	long			index		= -1;
	long			id			= -1;
	char			str[80];
	char			cmd[8];

	if (((CWinHcsStatusDoc *) GetDocument())->IsConnectedToHcs() == TRUE &&	pDispInfo->item.iItem == 0) {

		index	= pDispInfo->item.mask;
		pList->GetItemText(index, 0, str, 80);

		if (strlen(str) > 0) {

			id = (long) atol(str);

			if (id >= 0 && id < MAX_VARS) {

				((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables()->SetBeingChanged(id);

				cmd[0] = '!';
				cmd[1] = 0x24;
				cmd[2] = (char) id;

				((CWinHcsStatusDoc *) GetDocument())->SendMessageToHcs(cmd, 3);
			}
		}
	}
	
	*pResult = 0;
}

void CWinHcsStatusView::OnRclickVarList(NMHDR* pNMHDR, LRESULT* pResult) 
{
	CListCtrl		*pList		= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	LV_DISPINFO		*pDispInfo	= (LV_DISPINFO*)pNMHDR;
	long			index		= -1;
	long			id			= -1;
	BOOL			bExists		= FALSE;
	CMenu			PopMenu;
	CRect			rect;
	long			flags, x, y;
	CHcsIo			*pIO;
	char			str[80];

	if (pDispInfo->item.iItem == 0) {

		index	= pDispInfo->item.mask;
		pList->GetItemText(index, 0, str, 80);

		m_varListIdx = index;

		if (strlen(str) > 0) {

			id = (long) atol(str);

			if (id >= 0 && id < MAX_VARS) {

				pIO = ((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables();
				pIO->SetCurrentSelection(id);

				try {
					bExists = PopMenu.CreatePopupMenu();
					if (!bExists)
						AfxThrowUserException();

					if (((CWinHcsStatusDoc *) GetDocument())->IsConnectedToHcs() == FALSE)
						flags = MF_STRING | MF_DISABLED;
					else
						flags = MF_STRING | MF_ENABLED;

					// Add popup menu entries
					if (!PopMenu.AppendMenu(flags,	ID_COMMANDS_VARIABLE, _T("Set...")))
							AfxThrowUserException();


					flags = MF_STRING | MF_ENABLED;

					if (!PopMenu.AppendMenu(flags,	ID_COMMANDS_GETVARIABLE, _T("Add...")))
							AfxThrowUserException();

					if (!PopMenu.AppendMenu(flags,	ID_MENU_RMV_VAR, _T("Remove")))
							AfxThrowUserException();


					// Add display & track popup menus
					::GetWindowRect(pList->m_hWnd, rect);

					x = rect.left + (rect.right - rect.left) / 2;
					y = rect.top;

					if (!PopMenu.TrackPopupMenu(TPM_LEFTALIGN, x, y, GetParent()))
						AfxThrowUserException();
				}
				catch (CUserException* pEx)
				{
					MessageBeep(0);
					TRACE0("Warning: The popup couldn't be created!\n");

					pEx->Delete();
				}

				if (bExists)
					PopMenu.DestroyMenu();
			}
		}
	}

	*pResult = 0;
}


void CWinHcsStatusView::OnMenuRmvVar() 
{
	CListCtrl	*pList	= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);

	pList->DeleteItem(m_varListIdx);
}


void CWinHcsStatusView::RequestVariableUpdate() 
{
	CListCtrl	*pList	= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	long		n, i, id;
	char		buf[80];
	char		cmd[8];

	if (((CWinHcsStatusDoc *) GetDocument())->IsConnectedToHcs() == TRUE) {

		n = pList->GetItemCount();

		for (i = 0 ; i < n ; i++) {

			pList->GetItemText(i, 0, buf, 80);

			if (strlen(buf) > 0) {

				id = (long) atol(buf);

				if (id >= 0 && id < MAX_VARS) {

//					((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables()->SetBeingChanged(id);

					cmd[0] = '!';
					cmd[1] = 0x24;
					cmd[2] = (char) id;

					((CWinHcsStatusDoc *) GetDocument())->SendMessageToHcs(cmd, 3);
				}
			}
		}
	}
}


void CWinHcsStatusView::AddVariableDisplay(short id) 
{
	CListCtrl		*pList	= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	CHcsIo			*pIO	= ((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables();
	BOOL			bFound  = FALSE;
	long			n, i, id1;
	long			val;
	char			*sP;
	char			buf[80];
	char			cmd[8];

	// Look for duplicates
	n = pList->GetItemCount();

	for (i = 0 ; i < n && bFound == FALSE; i++) {

		pList->GetItemText(i, 0, buf, 80);

		if (strlen(buf) > 0) {

			id1 = (long) atol(buf);
			if (id == id1)
				bFound = TRUE;;
		}
	}

	// And add it
	if (bFound == FALSE) {

		sP = pIO->GetIoLabel(id);
		sprintf(buf, "%d: %s", id, sP);

		pList->InsertItem(0, buf);

		val = pIO->GetIdValue(id);
		sprintf(buf, _T("%d"), val);
		pList->SetItem(0, 1, LVIF_TEXT, buf, 0, 0, 0, 0);
	}

	if (((CWinHcsStatusDoc *) GetDocument())->IsConnectedToHcs() == TRUE) {

		((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables()->SetBeingChanged(id);

		cmd[0] = '!';
		cmd[1] = 0x24;
		cmd[2] = (char) id;

		((CWinHcsStatusDoc *) GetDocument())->SendMessageToHcs(cmd, 3);
	}
}


/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
//
//	Automation support
//
/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////

void CWinHcsStatusView::AutomationUpdate() 
{
	OnUpdate(NULL, 0, NULL);
}

void CWinHcsStatusView::AutomationClearConsole() 
{
	CListBox			*pList;

	pList = (CListBox*) GetDlgItem(IDC_CONSOLE);
	pList->ResetContent();
}

short CWinHcsStatusView::AutomationGetNetbitView() 
{
	return m_netbitDisplayType;
}

void CWinHcsStatusView::AutomationSetNetbitView(short nNewValue) 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();
	CButton			*pCBtn;

	m_netbitDisplayType = nNewValue;

	pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_MAN1);
	pCBtn->SetCheck(FALSE);
	pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_MAN2);
	pCBtn->SetCheck(FALSE);
	pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_DIO);
	pCBtn->SetCheck(FALSE);

	switch (m_netbitDisplayType) {
	default:
	case DISP_MAN1_4:
		pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_MAN1);
		pCBtn->SetCheck(TRUE);

		m_netbitDisplayType = DISP_MAN1_4;
		break;
	case DISP_MAN5_8:
		pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_MAN2);
		pCBtn->SetCheck(TRUE);
		break;
	case DISP_DIO:
		pCBtn = (CButton*)	GetDlgItem(IDC_RADIO_DIO);
		pCBtn->SetCheck(TRUE);
		break;
	}

	pApp->SetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_NETBIT, m_netbitDisplayType);

	OnUpdate(NULL, 0, NULL);
}


short CWinHcsStatusView::AutomationGetX10View() 
{
	return m_x10DisplayOfs / 16;
}

void CWinHcsStatusView::AutomationSetX10View(short nNewValue) 
{
	CWinHcsApp		*pApp  = (CWinHcsApp *) AfxGetApp();
	CComboBox		*pCBox = (CComboBox*)   GetDlgItem(IDC_COMBO_X10);

	if (nNewValue < 0 || nNewValue > 13)
		nNewValue = 0;

	m_x10DisplayOfs = nNewValue;
	pApp->SetProfileLong(IDS_REG_SETTINGS, IDS_REG_VIEW_X10, m_x10DisplayOfs);

	pCBox->SetCurSel(m_x10DisplayOfs);

	m_x10DisplayOfs = nNewValue * 16;
	OnUpdate(NULL, 0, NULL);
}


void CWinHcsStatusView::AutomationAddVariable(short id) 
{
	AddVariableDisplay(id);
}


void CWinHcsStatusView::AutomationRemoveVariable(short id) 
{
	CListCtrl	*pList	= (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);
	CHcsIo		*pIO	= ((CWinHcsStatusDoc *) GetDocument())->GetHcsVariables();
	int			index;
	LVFINDINFO	info;
	char		*sP;
	char		buf[80];

	sP = pIO->GetIoLabel(id);
	sprintf(buf, _T("%d: %s"), id, sP);

	info.flags	= LVFI_STRING;
	info.psz	= buf;

	index = pList->FindItem(&info, -1);
	pList->DeleteItem(index);
}

void CWinHcsStatusView::AutomationRemoveAllVariables() 
{
	CListCtrl		*pList = (CListCtrl*)	GetDlgItem(IDC_VAR_LIST);

	pList->DeleteAllItems();
}

BOOL CWinHcsStatusView::AutomationGetVisible() 
{
	return this->IsWindowVisible();
}

void CWinHcsStatusView::AutomationSetVisible(BOOL bNewValue) 
{
    CFrameWnd* pFrameWnd = this->GetParentFrame();

	if (bNewValue == TRUE) {

        if (!pFrameWnd->IsWindowVisible())
            pFrameWnd->ShowWindow(SW_SHOWNA);
        
        pFrameWnd = pFrameWnd->GetParentFrame() ;                
        if (pFrameWnd && !pFrameWnd->IsWindowVisible())
            pFrameWnd->ShowWindow(SW_SHOWNA);

		OnUpdate(NULL, 0, NULL);

	} else {

		pFrameWnd->ShowWindow(SW_HIDE);
	}

	AfxOleSetUserCtrl(bNewValue);
}


BOOL CWinHcsStatusView::AutomationMaximize() 
{
    CFrameWnd* pFrameWnd = this->GetParentFrame();
	return pFrameWnd->ShowWindow( SW_SHOWMAXIMIZED ) ;
}
BOOL CWinHcsStatusView::AutomationMinimize() 
{
    CFrameWnd* pFrameWnd = this->GetParentFrame();
	return pFrameWnd->ShowWindow( SW_MINIMIZE ) ;
}
BOOL CWinHcsStatusView::AutomationRestore() 
{
    CFrameWnd* pFrameWnd = this->GetParentFrame();
	return pFrameWnd->ShowWindow( SW_RESTORE ) ;
}
