/* ====================================================================
 *
 * WinHcs
 * Copyright (c) 1997-1999. Philippe Printz.		All rights reserved.
 *
 *	The WinHcs software is OSI Certified Open Source Software.
 *	OSI Certified is a certification mark of the Open Source Initiative.
 *
 *	WinHcs maybe distributed under the terms of the	GNU General Public License.
 *
 *
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *    software must display the following acknowledgment:
 *    "This product includes software developed by Philippe Printz
 *    for use in the HCS project."
 *
 * 4. The names "WinHcs" and "HCS" must not be used to
 *    endorse or promote products derived from this software without
 *    prior written permission.
 *
 * 5. Products derived from this software may not be called "WinHcs"
 *    nor may "WinHcs" appear in their names without prior written
 *    permission of the WinHcs.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Philippe Printz
 *    for use in the HCS project."
 *
 * THIS SOFTWARE IS PROVIDED BY PHILIPPE PRINTZ ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL PHILIPPE PRINTZ OR
 * OTHER CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals and was originally based on software written by
 * Philippe C. Printz.
 *
 */

// WinHcsStatusDoc.cpp : implementation file
//

#include "stdafx.h"
#include "WinHcs.h"
#include "WinHcsStatusDoc.h"
#include "WinHcsStatusView.h"
#include "HcsIo.h"

#include "ListenSocket.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#include <comdef.h>


long	_logTimerCount = 0;
long	_storeLogTimerCount = 0;

/////////////////////////////////////////////////////////////////////////////
// HCS log timer proc
void CALLBACK EXPORT LogTimerProc(
   HWND hWnd,      // handle of CWnd that called SetTimer
   UINT nMsg,      // WM_TIMER
   UINT nIDEvent,   // timer identification
   DWORD dwTime    // system time
)
{
	CWinHcsApp			*pApp	= (CWinHcsApp *) AfxGetApp();
	CMainFrame			*pFrame = (CMainFrame*) AfxGetMainWnd();
	CWinHcsStatusDoc	*pDoc	= pApp->GetHcsStatusDoc();
	CWinHcsStatusView	*pView;
	long				i;
	POSITION			pos;
	BOOL				bVal, bAutoUpload;
	static	long		last_logNbEntries = -1;
	
	if (pDoc->IsConnectedToHcs()) {

		pDoc->SendMessageToHcs("!\x08", 2);

		pos = pDoc->GetFirstViewPosition();

		while (pos != NULL) {

			pView = (CWinHcsStatusView *) pDoc->GetNextView(pos);
			pView->RequestVariableUpdate();
		}   
	}

	// Auto upload?
	pDoc->GetLogUpdateSettings(&i, &bAutoUpload, &bVal);

	if (pDoc->IsConnectedToHcs() && bAutoUpload == TRUE && last_logNbEntries != pDoc->m_logNbEntries) {

		last_logNbEntries = pDoc->m_logNbEntries;
		pApp->OnCommandsLogdataGet();

	} else {

		last_logNbEntries = -1;
	}


	// And setup timer for next one
	i = i * 1000 * 60;
	if (i == 0)	i = 1;

	(void) pFrame->SetTimer(ID_TIMER_LOG, i, LogTimerProc);

	_logTimerCount++;
}


/////////////////////////////////////////////////////////////////////////////
// HCS store log timer proc
void CALLBACK EXPORT StoreLogTimerProc(
   HWND hWnd,      // handle of CWnd that called SetTimer
   UINT nMsg,      // WM_TIMER
   UINT nIDEvent,   // timer identification
   DWORD dwTime    // system time
)
{
	CWinHcsApp			*pApp	= (CWinHcsApp *) AfxGetApp();
	CMainFrame			*pFrame = (CMainFrame*) AfxGetMainWnd();
	CWinHcsStatusDoc	*pDoc	= pApp->GetHcsStatusDoc();
	long				i;

	if (pDoc->IsConnectedToHcs() && pDoc->m_OdbcEnable == TRUE && pDoc->m_OdbcUpdAtConnect == TRUE)
		pApp->OnCommandsLogdataStorelogbuffer();

	i = pDoc->m_OdbcUpdateFreq * 1000 * 60;
	if (i == 0)	i = 1;

	(void) pFrame->SetTimer(ID_TIMER_STORELOG, i, StoreLogTimerProc);

	_storeLogTimerCount++;
}

/////////////////////////////////////////////////////////////////////////////
// HCS receiving thread

UINT CommWatchProc( LPVOID pParam )
{
	CWinHcsApp			*pApp	= (CWinHcsApp *) AfxGetApp();
	CMainFrame			*pFrame = (CMainFrame*) AfxGetMainWnd();
	CWinHcsStatusDoc	*pDoc	= pApp->GetHcsStatusDoc();

	if (pDoc != NULL)
		pDoc->ReceiveAndProcessHcsMessages(pFrame);

	return 0;
}



/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusDoc

IMPLEMENT_DYNCREATE(CWinHcsStatusDoc, CDocument)

CWinHcsStatusDoc::CWinHcsStatusDoc()
{
	CWinHcsApp	*pApp = (CWinHcsApp *) AfxGetApp();
	CMainFrame	*pFrame = (CMainFrame*) AfxGetMainWnd();
	long		i, j, n, m;
	CString		port, modemInitCmd;
	BOOL		bVal, bVal1;

	m_OdbcEnable			= FALSE;

	m_replyP				= NULL;
	m_replySz				= 0;

	m_logPos				= 0;
	m_logDataSz				= 0;
	memset(m_logData, 0xff, LOG_BUF_SZ);

	m_logReceived			= FALSE;
	m_logSizeReceived		= FALSE;
	m_logNbEntries			= 0;

	// Do not destroy the document on close
	m_bAutoDelete			= FALSE;

	// initialize com structure
	m_UsingModem			= FALSE;
	m_CallerIDModem			= FALSE;

	m_rs232_in_state		= RS232_NOT_ACTIVE;
	m_rs232_out_state		= RS232_NOT_ACTIVE;

	m_idComDev				= 0;
	m_fConnected			= FALSE;
	m_bPort					= 1;
	m_dwBaudRate			= CBR_9600;
	m_bByteSize				= 8;
	m_bFlowCtrl				= FC_RTSCTS;
//	m_bFlowCtrl				= FC_XONXOFF;
	m_bParity				= NOPARITY;
	m_bStopBits				= ONESTOPBIT;
	m_fXonXoff				= FALSE;



	// Retrieve from registry
	port			= pApp->GetProfileCString(	IDS_REG_SETTINGS,	IDS_REG_COM,		"COM1");
//	bVal			= pApp->GetProfileLong(		IDS_REG_SETTINGS,	IDS_REG_MODEM,		FALSE);
	bVal = FALSE;
	modemInitCmd	= pApp->GetProfileCString(	IDS_REG_SETTINGS,	IDS_REG_MODEM_INIT, NULL);
	SetRS232Settings(port, bVal, modemInitCmd);

	m_pListenSocket	= NULL;
	m_listenPort	= pApp->GetProfileLong(		IDS_REG_SETTINGS,	IDS_REG_TCP_PORT,	10000);

	m_AutoConnectAtStartup	= pApp->GetProfileLong(IDS_REG_SETTINGS,	IDS_REG_AUTOCONNECT,FALSE);
	m_ShowViewAtStartup		= pApp->GetProfileLong(IDS_REG_SETTINGS,	IDS_REG_AUTOVIEW,	FALSE);

	m_OdbcEnable		= pApp->GetProfileLong(	 IDS_REG_ODBC, IDS_REG_ENABLED,			FALSE);
	m_OdbcDSN			= pApp->GetProfileCString(IDS_REG_ODBC,IDS_REG_DSN,				NULL);
	m_OdbcTable			= pApp->GetProfileCString(IDS_REG_ODBC,IDS_REG_TABLE,			"LoggedData");
	m_OdbcUpdAtConnect	= pApp->GetProfileLong(	 IDS_REG_ODBC, IDS_REG_UPD_CONNECT,		FALSE);
	m_OdbcUpdateFreq	= pApp->GetProfileLong(	 IDS_REG_ODBC, IDS_REG_UPD_RATE,		60);

	i = m_OdbcUpdateFreq * 1000 * 60;
	(void) pFrame->SetTimer(ID_TIMER_STORELOG, i, StoreLogTimerProc);

	//
	memset( &m_osRead,  0, sizeof(OVERLAPPED));
	memset( &m_osWrite, 0, sizeof(OVERLAPPED));

	// create I/O event used for overlapped reads / writes
	m_osRead.hEvent = CreateEvent(	NULL,		// no security
									TRUE,		// explicit reset req
									FALSE,		// initial event reset
									NULL );		// no name
	ASSERT(m_osRead.hEvent != NULL);

	m_osWrite.hEvent = CreateEvent(	NULL,		// no security
									TRUE,		// explicit reset req
									FALSE,		// initial event reset
									NULL );		// no name
	if (NULL == m_osWrite.hEvent) {

		CloseHandle( m_osRead.hEvent );
		ASSERT(m_osWrite.hEvent != NULL);
	}


	// Basic HCS configuration
	i				= pApp->GetProfileLong(	IDS_REG_HCS,	IDS_REG_TIME_UPD_RATE,		TIME_UPDATE_FAST);
	SetTimeUpdateRate(i);

	i				= pApp->GetProfileLong(	IDS_REG_HCS,	IDS_REG_LOG_UPD_RATE,		10);
	bVal1			= pApp->GetProfileLong(	IDS_REG_HCS,	IDS_REG_LOG_UPD_AUTO,		FALSE);
	bVal			= pApp->GetProfileLong(	IDS_REG_HCS,	IDS_REG_LOG_CLEAR,			FALSE);
	SetLogUpdateSettings(i, bVal1, bVal);

	i = m_logUpdateRate * 1000 * 60;
	(void) pFrame->SetTimer(ID_TIMER_LOG, i, LogTimerProc);



	m_Modules_PL.Setup(		IoChar,		MAX_MODULES_PL,		NULL);
	m_Modules_MCIR.Setup(	IoChar,		MAX_MODULES_MCIR,	NULL);
	m_Modules_LCD.Setup(	IoChar,		MAX_MODULES_LCD,	NULL);
	m_Modules_DIO.Setup(	IoChar,		MAX_MODULES_DIO,	NULL);
	m_Modules_AMAN.Setup(	IoChar,		MAX_MODULES_MAN,	NULL);
	m_HcsInputs.Setup(		IoBool,		MAX_HCS_INPUTS,		_T("HCS Inputs"));
	m_HcsOutputs.Setup(		IoBool,		MAX_HCS_OUTPUTS,	_T("HCS Ouputs"));
	m_Netbits.Setup(		IoBool,		MAX_NETBITS,		_T("HCS Netbits"));
	m_ADCs.Setup(			IoShort,	MAX_ADCS,			_T("HCS ADCs"));
	m_DACs.Setup(			IoShort,	MAX_DACS,			_T("HCS DACs"));
	m_X10.Setup(			IoX10,		MAX_X10,			_T("HCS X10"));
	m_XPRESS_Vars.Setup(	IoShort,	MAX_VARS,			_T("HCS Variables"));
	m_XPRESS_Timers.Setup(	IoShort,	MAX_TIMERS,			_T("HCS Timers"));
	m_XPRESS_Logs.Setup(	IoShort,	MAX_LOGID,			_T("HCS Log IDs"));


	m_Modules_PL.SetInUse(0, _T(""));


	m_Module_BUFIO = pApp->GetProfileLong(IDS_REG_HCS,	IDS_REG_NB_BUFIO,  0);
	if (m_Module_BUFIO < 0 || m_Module_BUFIO > 1)
		m_Module_BUFIO = 0;

	n = 16 + m_Module_BUFIO * 24;
	for (i = 0 ; i < 40 ; i++) {

		if (i < n) {

			m_HcsInputs.SetInUse(i);

			if (i < 13 || i > 15)
				m_HcsOutputs.SetInUse(i);
			else
				m_HcsOutputs.SetNotInUse(i);

		} else {

			m_HcsInputs.SetNotInUse(i);
			m_HcsOutputs.SetNotInUse(i);
		}

		if (i < 8)
			m_ADCs.SetInUse(i);
	}


	n = pApp->GetProfileLong(IDS_REG_HCS,	IDS_REG_NB_MCIR, 0);
	for (i = 0 ; i < MAX_MODULES_MCIR ; i++) {
		if (i < n)			m_Modules_MCIR.SetInUse(i);
		else				m_Modules_MCIR.SetNotInUse(i);
	}

	n = pApp->GetProfileLong(IDS_REG_HCS,	IDS_REG_NB_LCD,  0);
	for (i = 0 ; i < MAX_MODULES_LCD ; i++) {
		if (i < n)			m_Modules_LCD.SetInUse(i);
		else				m_Modules_LCD.SetNotInUse(i);
	}

	n = pApp->GetProfileLong(IDS_REG_HCS,	IDS_REG_NB_DIO,  0);
	for (i = 0 ; i < MAX_MODULES_DIO ; i++) {
		if (i < n)			m_Modules_DIO.SetInUse(i);
		else				m_Modules_DIO.SetNotInUse(i);
	}
	n = n * 8;
	for (i = 0 ; i < 64; i++) {
		if (i < n)			m_Netbits.SetInUse(i);
		else				m_Netbits.SetNotInUse(i);
	}


	n = pApp->GetProfileLong(IDS_REG_HCS,	IDS_REG_NB_AMAN, 0);
	for (i = 0 ; i < MAX_MODULES_MAN ; i++) {
		if (i < n) {
			m_Modules_AMAN.SetInUse(i);

			m = 96 + 16 * i;
			for (j = m ; j < (m + 16) ; j++)
				m_Netbits.SetInUse(j);

			m = 16 + 8 * i;
			for (j = m ; j < (m + 6) ; j++)
				m_ADCs.SetInUse(j);

			m = 4 * i;
			m_DACs.SetInUse(m);
			m_DACs.SetInUse(m + 1);

		} else {

			m_Modules_AMAN.SetNotInUse(i);

			m = 96 + 16 * i;
			for (j = m ; j < (m + 16) ; j++)
				m_Netbits.SetNotInUse(j);

			m = 16 + 8 * i;
			for (j = m ; j < (m + 6) ; j++)
				m_ADCs.SetNotInUse(j);

			m = 4 * i;
			m_DACs.SetNotInUse(m);
			m_DACs.SetNotInUse(m + 1);
		}
	}

	memset( m_HcsTime,		 0, 12);
	m_HcsTime[6] = 1;
	m_HcsTime[5] = 1;


	m_ConsoleMessages.RemoveAll();
	m_HcsCommands.RemoveAll();

	m_prgSizeToSend	= 0;
	m_prgDataP		= NULL;

	EnableAutomation();

	// And know setup listening socket
	m_pListenSocket = new CListenSocket();
	 
	if ( m_pListenSocket ) {

		BOOL	ret = FALSE;

		if ( m_pListenSocket->Create( m_listenPort, SOCK_STREAM, FD_ACCEPT ) )
			ret = m_pListenSocket->Listen();

		if (ret == FALSE) {

			CString strMsg;

			int nErr = m_pListenSocket->GetLastError();
			if ( nErr == WSAEADDRINUSE )
				strMsg.Format( "Listening port %d is in use.", m_listenPort );
			else
				strMsg.Format( "Error while opening listening port %d.", m_listenPort );

			AfxMessageBox( strMsg, MB_OK|MB_ICONSTOP );

			m_pListenSocket->Close();

			delete m_pListenSocket;
			m_pListenSocket = NULL;
		}
	}
	else
		AfxMessageBox( "Cannot listen on socket.", MB_OK|MB_ICONSTOP );

}

BOOL CWinHcsStatusDoc::OnNewDocument()
{

	if (!CDocument::OnNewDocument())
		return FALSE;


	return TRUE;
}


CWinHcsStatusDoc::~CWinHcsStatusDoc()
{
	CWinHcsApp	*pApp = (CWinHcsApp *) AfxGetApp();

	delete m_pListenSocket;

	CloseConnectionToHcs();

	CloseHandle( m_osRead.hEvent );
	CloseHandle( m_osWrite.hEvent );

	pApp->SetProfileLong(		IDS_REG_SETTINGS,	IDS_REG_TCP_PORT,	m_listenPort);

	pApp->SetProfileLong(		IDS_REG_SETTINGS,	IDS_REG_AUTOCONNECT,m_AutoConnectAtStartup);
	pApp->SetProfileLong(		IDS_REG_SETTINGS,	IDS_REG_AUTOVIEW,	m_ShowViewAtStartup);

	pApp->SetProfileLong(		IDS_REG_ODBC,	IDS_REG_ENABLED,		m_OdbcEnable);
	pApp->SetProfileCString(	IDS_REG_ODBC,	IDS_REG_DSN,			m_OdbcDSN.GetBuffer(256));
	pApp->SetProfileCString(	IDS_REG_ODBC,	IDS_REG_TABLE,			m_OdbcTable.GetBuffer(256));
	pApp->SetProfileLong(		IDS_REG_ODBC,	IDS_REG_UPD_CONNECT,	m_OdbcUpdAtConnect);
	pApp->SetProfileLong(		IDS_REG_ODBC,	IDS_REG_UPD_RATE,		m_OdbcUpdateFreq);

	pApp->SetProfileLong(		IDS_REG_HCS,	IDS_REG_NB_BUFIO,		m_Module_BUFIO);
	pApp->SetProfileLong(		IDS_REG_HCS,	IDS_REG_NB_MCIR,		m_Modules_MCIR.GetMaxInGroup());
	pApp->SetProfileLong(		IDS_REG_HCS,	IDS_REG_NB_LCD,			m_Modules_LCD.GetMaxInGroup());
	pApp->SetProfileLong(		IDS_REG_HCS,	IDS_REG_NB_DIO,			m_Modules_DIO.GetMaxInGroup());
	pApp->SetProfileLong(		IDS_REG_HCS,	IDS_REG_NB_AMAN,		m_Modules_AMAN.GetMaxInGroup());

	m_HcsInputs.SaveSetup();
	m_HcsOutputs.SaveSetup();
	m_Netbits.SaveSetup();
	m_ADCs.SaveSetup();
	m_DACs.SaveSetup();
	m_X10.SaveSetup();
	m_XPRESS_Vars.SaveSetup();
	m_XPRESS_Timers.SaveSetup();
	m_XPRESS_Logs.SaveSetup();
}

void CWinHcsStatusDoc::OnFinalRelease()
{
	// When the last reference for an automation object is released
	// OnFinalRelease is called.  The base class will automatically
	// deletes the object.  Add additional cleanup required for your
	// object before calling the base class.

	CDocument::OnFinalRelease();
}


BEGIN_MESSAGE_MAP(CWinHcsStatusDoc, CDocument)
	//{{AFX_MSG_MAP(CWinHcsStatusDoc)
		// NOTE - the ClassWizard will add and remove mapping macros here.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

BEGIN_DISPATCH_MAP(CWinHcsStatusDoc, CDocument)
	//{{AFX_DISPATCH_MAP(CWinHcsStatusDoc)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogID", m_hcsLogID, VT_I4)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogValue", m_hcsLogValue, VT_I4)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogYear", m_hcsLogYear, VT_I2)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogMonth", m_hcsLogMonth, VT_I2)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogDay", m_hcsLogDay, VT_I2)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogHour", m_hcsLogHour, VT_I2)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogMinute", m_hcsLogMinute, VT_I2)
	DISP_PROPERTY(CWinHcsStatusDoc, "hcsLogSecond", m_hcsLogSecond, VT_I2)
	DISP_PROPERTY_EX(CWinHcsStatusDoc, "IsConnected", AutomationGetIsConnected, AutomationSetIsConnected, VT_BOOL)
	DISP_PROPERTY_EX(CWinHcsStatusDoc, "newConsoleMsg", AutomationGetNewConsoleMsg, SetNotSupported, VT_BOOL)
	DISP_PROPERTY_EX(CWinHcsStatusDoc, "NbLogEntries", AutomationGetNbLogEntries, SetNotSupported, VT_I4)
	DISP_PROPERTY_EX(CWinHcsStatusDoc, "lastConsoleMsg", AutomationGetLastConsoleMsg, SetNotSupported, VT_BSTR)
	DISP_PROPERTY_EX(CWinHcsStatusDoc, "Application", AutomationGetApplication, SetNotSupported, VT_VARIANT)
	DISP_PROPERTY_EX(CWinHcsStatusDoc, "Window", AutomationGetWindow, SetNotSupported, VT_VARIANT)
	DISP_FUNCTION(CWinHcsStatusDoc, "AddLogEntry", AutomationAddLogEntry, VT_EMPTY, VTS_I2 VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "ClearLogEntries", AutomationClearLogEntries, VT_EMPTY, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetLogData", AutomationGetLogData, VT_BOOL, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetFirstLogEntry", AutomationGetFirstLogEntry, VT_BOOL, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetNextLogEntry", AutomationGetNextLogEntry, VT_BOOL, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetTime", AutomationSetTime, VT_EMPTY, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetTime", AutomationGetTime, VT_DATE, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "Speak", AutomationSpeak, VT_EMPTY, VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SendToNetwork", AutomationSendToNetwork, VT_EMPTY, VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetDAC", AutomationSetDac, VT_EMPTY, VTS_I2 VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetADC", AutomationSetADC, VT_EMPTY, VTS_I2 VTS_BOOL VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetOutput", AutomationSetOutput, VT_EMPTY, VTS_I2 VTS_BOOL)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetInput", AutomationSetInput, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BOOL)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetNetbit", AutomationSetNetbit, VT_EMPTY, VTS_I2 VTS_BOOL)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetVariable", AutomationSetVariable, VT_EMPTY, VTS_I2 VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetX10OnOrOff", AutomationSetX10OnOrOff, VT_EMPTY, VTS_I2 VTS_BOOL)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetX10DimOrBright", AutomationSetX10DimOrBright, VT_EMPTY, VTS_I2 VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetX10AllOff", AutomationSetX10AllOff, VT_EMPTY, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetX10AllOn", AutomationSetX10AllOn, VT_EMPTY, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetADC", AutomationGetADC, VT_I2, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetDAC", AutomationGetDAC, VT_I2, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetOutput", AutomationGetOutput, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetInput", AutomationGetInput, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetNetbit", AutomationGetNetbit, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetVariable", AutomationGetVariable, VT_I2, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetX10", AutomationGetX10, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetADCProperties", AutomationSetADCProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetDACProperties", AutomationSetDACProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetOutputProperties", AutomationSetOutputProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetInputProperties", AutomationSetInputProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetNetbitProperties", AutomationSetNetbitProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetVariableProperties", AutomationSetVariableProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetX10Properties", AutomationSetX10Properties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetLogIDProperties", AutomationSetLogIDProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "SetTimerProperties", AutomationSetTimerProperties, VT_EMPTY, VTS_I2 VTS_BOOL VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsADCInUse", AutomationIsADCInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsDACInUse", AutomationIsDACInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsOutputInUse", AutomationIsOutputInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsInputInUse", AutomationIsInputInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsNetBitInUse", AutomationIsNetBitInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsVariableInUse", AutomationIsVariableInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsX10InUse", AutomationIsX10InUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsLogIdInUse", AutomationIsLogIdInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "IsTimerInUse", AutomationIsTimerInUse, VT_BOOL, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetADCLabel", AutomationGetADCLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetDACLabel", AutomationGetDACLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetOutputLabel", AutomationGetOutputLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetInputLabel", AutomationGetInputLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetNetbitLabel", AutomationGetNetbitLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetVariableLabel", AutomationGetVariableLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetX10Label", AutomationGetX10Label, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetLogIDLabel", AutomationGetLogIDLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetTimerLabel", AutomationGetTimerLabel, VT_BSTR, VTS_I2)
	DISP_FUNCTION(CWinHcsStatusDoc, "AddWindow", AutomationAddWindow, VT_EMPTY, VTS_NONE)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetFirstDBLogEntry", AutomationGetFirstDBLogEntry, VT_BOOL, VTS_I4 VTS_I4 VTS_BSTR VTS_BSTR)
	DISP_FUNCTION(CWinHcsStatusDoc, "GetNextDBLogEntry", AutomationGetDBNextLogEntry, VT_BOOL, VTS_NONE)
	//}}AFX_DISPATCH_MAP
END_DISPATCH_MAP()

// Note: we add support for IID_IWinHcsStatusDoc to support typesafe binding
//  from VBA.  This IID must match the GUID that is attached to the 
//  dispinterface in the .ODL file.

// {BF76971B-4DAC-11D3-B89C-0000861E056C}
static const IID IID_IWinHcsStatusDoc =
{ 0xbf76971b, 0x4dac, 0x11d3, { 0xb8, 0x9c, 0x0, 0x0, 0x86, 0x1e, 0x5, 0x6c } };

BEGIN_INTERFACE_MAP(CWinHcsStatusDoc, CDocument)
	INTERFACE_PART(CWinHcsStatusDoc, IID_IWinHcsStatusDoc, Dispatch)
END_INTERFACE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusDoc diagnostics

#ifdef _DEBUG
void CWinHcsStatusDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CWinHcsStatusDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusDoc serialization

void CWinHcsStatusDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// TODO: add storing code here
	}
	else
	{
		// TODO: add loading code here
	}
}

/////////////////////////////////////////////////////////////////////////////
// CWinHcsStatusDoc commands

// Low level HCS I/O routines
BOOL CWinHcsStatusDoc::OpenConnectionToHcs(void)
{
	CWinHcsApp			*pApp = (CWinHcsApp *) AfxGetApp();
	CMainFrame			*pFrame = (CMainFrame*) AfxGetMainWnd();
	CWinHcsStatusView	*pView;
	POSITION			pos;
	COMMTIMEOUTS		CommTimeOuts;
	CWinThread			*pThread;
	long				n, nbAMAN;
	unsigned char		mask;
	char				szPort[16];
	char				cmd[16];

	pFrame->SetHcsConnectionStatus( HcsStatusDialing );

	// Define COM path
	if (m_bPort < 1 || m_bPort > MAXPORTS) {

	  pFrame->SetHcsConnectionStatus( HcsStatusDisconnected );
	  return FALSE;
	}

	sprintf(szPort, "COM%d", m_bPort );

	// open COMM device
	if ((m_idComDev = CreateFile(
				szPort,
				GENERIC_READ | GENERIC_WRITE,
				0,												// exclusive access
				NULL,											// no security attrs
				OPEN_EXISTING,
				FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,	// overlapped I/O
				NULL )) == (HANDLE) -1 ) {

	  pFrame->SetHcsConnectionStatus( HcsStatusDisconnected );
	  return FALSE;
	}

	// setup device buffers
	SetupComm( m_idComDev, RXQUEUE, TXQUEUE);

	// get any early notifications
	SetCommMask( m_idComDev, EV_RXCHAR);

	// purge any information in the buffer
	PurgeComm( m_idComDev, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR );

	// set up for overlapped I/O
	CommTimeOuts.ReadIntervalTimeout		= 0xFFFFFFFF;
	CommTimeOuts.ReadTotalTimeoutMultiplier = 0;
	CommTimeOuts.ReadTotalTimeoutConstant	= 1000;
	CommTimeOuts.WriteTotalTimeoutMultiplier= 0;
	CommTimeOuts.WriteTotalTimeoutConstant	= 1000;

	SetCommTimeouts( m_idComDev, &CommTimeOuts );

	m_fConnected = TRUE;

	// setup comm
	if (SetupConnectionToHcs() == FALSE) {

		m_fConnected = FALSE;
		CloseHandle( m_idComDev );
		pFrame->SetHcsConnectionStatus( HcsStatusDisconnected );

		return FALSE;
	}

	// Create a secondary thread to watch for an event.
	if (NULL == (pThread = AfxBeginThread(CommWatchProc, NULL))) {

		m_fConnected = FALSE;
		CloseHandle( m_idComDev );
		pFrame->SetHcsConnectionStatus( HcsStatusDisconnected );

		return FALSE;
	}

	m_dwThreadID = pThread->m_nThreadID; /*dwThreadID */;
	m_hWatchThread = pThread->m_hThread;

	// assert DTR
	EscapeCommFunction( m_idComDev, SETDTR );

	pFrame->SetHcsConnectionStatus( HcsStatusConnected );


	// And now setup initial commands (Time refresh rate & get log size)
	SetTimeUpdateRate(m_HcsUpdatePeriod);
	SendMessageToHcs("!\x08", 2);


	// Set ADC and DAC values to send depending on Nb of AMAN and DIO
	cmd[0] = '!';					// ADC bitmap
	cmd[1] = 0x19;
	nbAMAN = m_Modules_AMAN.GetMaxInGroup();
	mask = 0xff;
	if (nbAMAN < 7) {
		n = 8 - 2 - nbAMAN;
		mask >>= n;
	}
	cmd[2] = (char) mask;
	n = 8 - nbAMAN;
	mask = 0x03 >> n;
	cmd[3] = (char) mask;
	cmd[4] = (char) 0;
	SendMessageToHcs(cmd, 5);

	cmd[0] = '!';					// DAC bitmap
	cmd[1] = 0x1C;
	n = (8 - nbAMAN) / 2;
	mask = 0x0f >> n;
	cmd[2] = (char) mask;
	SendMessageToHcs(cmd, 3);

	pos = GetFirstViewPosition();

	while (pos != NULL) {
		pView = (CWinHcsStatusView *) GetNextView(pos);
		pView->RequestVariableUpdate();
	}   


	_SOUND(m_sndHcsConnect);

	// Get log if enabled
	if (m_OdbcEnable == TRUE && m_OdbcUpdAtConnect == TRUE)
		pApp->OnCommandsLogdataStorelogbuffer();

	return TRUE;
}


BOOL CWinHcsStatusDoc::CloseConnectionToHcs(void)
{
	CMainFrame		*pFrame = (CMainFrame*) AfxGetMainWnd();
	CWinHcsApp		*pApp = (CWinHcsApp *) AfxGetApp();
	long			i;

	// Ensure we are connected
	if (m_fConnected == FALSE)
		return FALSE;

	// set m_fConnected flag to FALSE
	m_fConnected = FALSE;

	// disable event notification and wait for thread to halt
	SetCommMask( m_idComDev, 0 );

	// block until thread has been halted
	i = 1000;
	while(m_dwThreadID != 0 && i > 0) {
		Sleep(10);
		i--;
	}

	// drop DTR
	EscapeCommFunction( m_idComDev, CLRDTR );

	// purge any outstanding reads/writes and close device handle
	PurgeComm( m_idComDev, PURGE_TXABORT | PURGE_RXABORT | PURGE_TXCLEAR | PURGE_RXCLEAR );
	CloseHandle( m_idComDev );

	if (pFrame != NULL)
		pFrame->SetHcsConnectionStatus( HcsStatusDisconnected );

	UpdateAllViews(NULL);

	return TRUE;
}

BOOL CWinHcsStatusDoc::IsConnectedToHcs(void)
{
	return m_fConnected;
}


BOOL CWinHcsStatusDoc::SetupConnectionToHcs(void)
{
	BYTE		bSet;
	DCB			dcb;

	// Ensure we are connected
	if (m_fConnected == FALSE)
		return FALSE;

	// Retrieve current settings
	dcb.DCBlength = sizeof( DCB );
	GetCommState( m_idComDev, &dcb );

	dcb.BaudRate		= m_dwBaudRate;
	dcb.ByteSize		= m_bByteSize;
	dcb.Parity			= m_bParity;
	dcb.StopBits		= m_bStopBits;

	// setup hardware flow control
	bSet				= (BYTE) ((m_bFlowCtrl & FC_DTRDSR) != 0);
	dcb.fOutxDsrFlow	= bSet;

	if (bSet)
		dcb.fDtrControl = DTR_CONTROL_HANDSHAKE;
	else
		dcb.fDtrControl = DTR_CONTROL_ENABLE;

	bSet				= (BYTE) ((m_bFlowCtrl & FC_RTSCTS) != 0);
	dcb.fOutxCtsFlow	= bSet;

	if (bSet)
		dcb.fRtsControl = RTS_CONTROL_HANDSHAKE;
	else
		dcb.fRtsControl = RTS_CONTROL_ENABLE;

	// setup software flow control
	bSet				= (BYTE) ((m_bFlowCtrl & FC_XONXOFF) != 0);

	dcb.fInX			= bSet;
	dcb.fOutX			= bSet;
	dcb.XonChar			= ASCII_XON;
	dcb.XoffChar		= ASCII_XOFF;
	dcb.XonLim			= 100;
	dcb.XoffLim			= 100;

	// other various settings
	dcb.fBinary			= TRUE;
	dcb.fParity			= TRUE;

	return SetCommState( m_idComDev, &dcb );
}


long CWinHcsStatusDoc::ReadBlockFromHcs(char *pData, unsigned long sz)
{
	BOOL				fReadStat;
	COMSTAT				ComStat;
	unsigned long		dwErrorFlags;
	unsigned long		dwLength;
	DWORD				dwError;

	// Ensure we are connected
	if (m_fConnected == FALSE) {

		_SOUND(m_sndHcsError);
		AfxMessageBox(IDS_ERR_NOT_CONNECTED, MB_OK, 0);
		return 0;
	}

	// only try to read number of bytes in queue 
	ClearCommError( m_idComDev, &dwErrorFlags, &ComStat );
	if (sz < ComStat.cbInQue)		dwLength = sz;
	else							dwLength = ComStat.cbInQue;

	if (dwLength > 0)	{

		fReadStat = ReadFile( m_idComDev, pData, dwLength, &dwLength, &m_osRead );

		if (!fReadStat)	{

			if (GetLastError() == ERROR_IO_PENDING)	{
	
				// We have to wait for read to complete. 
				// This function will timeout according to the
				// CommTimeOuts.ReadTotalTimeoutConstant variable
				// Every time it times out, check for m_bPort errors
		
				while(!GetOverlappedResult( m_idComDev, &m_osRead, &dwLength, TRUE)) {

					dwError = GetLastError();
					if(dwError == ERROR_IO_INCOMPLETE)
						continue;			// normal result if not finished
					else {
						// an error occurred, try to recover
						ClearCommError( m_idComDev, &dwErrorFlags, &ComStat);
						break;
					}			
				}

			}	else {

				CWinHcsStatusView	*pView;
				POSITION			pos;

				m_rs232_in_state = RS232_ERROR;
				if ((pos = GetFirstViewPosition()) != NULL && (pView = (CWinHcsStatusView *) GetNextView(pos)) != NULL)
					pView->ShowRS232State();
	
				// some other error occurred
				dwLength = 0;
				ClearCommError( m_idComDev, &dwErrorFlags, &ComStat );

				return dwLength;
			}
		}
	}

	return dwLength;
}


BOOL CWinHcsStatusDoc::WriteBlockToHcs(char *pData, unsigned long sz)
{
	CMainFrame			*pFrame = (CMainFrame*) AfxGetMainWnd();
	CWinHcsStatusView	*pView  = NULL;
	POSITION			pos		= NULL;
	BOOL				fWriteStat;
	DWORD				dwBytesWritten;
	DWORD				dwErrorFlags;
	DWORD   			dwError;
	COMSTAT				ComStat;

	// Ensure we are connected
	if (m_fConnected == FALSE) {

		_SOUND(m_sndHcsError);
		AfxMessageBox(IDS_ERR_NOT_CONNECTED , MB_OK, 0);
		return FALSE;
	}

	m_rs232_out_state = RS232_ACTIVE;
	if ((pos = GetFirstViewPosition()) != NULL && (pView = (CWinHcsStatusView *) GetNextView(pos)) != NULL)
		pView->ShowRS232State();

	fWriteStat = WriteFile( m_idComDev, pData, sz, &sz, &m_osWrite );

	// Note that normally the code will not execute the following
	// because the driver caches write operations. Small I/O requests 
	// (up to several thousand bytes) will normally be accepted 
	// immediately and WriteFile will return true even though an
	// overlapped operation was specified

	if (!fWriteStat) {

		if (GetLastError() == ERROR_IO_PENDING)	{

			// We should wait for the completion of the write operation
			// so we know if it worked or not

			// This is only one way to do this. It might be beneficial to 
			// the to place the writing operation in a separate thread 
			// so that blocking on completion will not negatively 
			// affect the responsiveness of the UI

			// If the write takes long enough to complete, this 
			// function will timeout according to the
			// CommTimeOuts.WriteTotalTimeoutConstant variable.
			// At that time we can check for errors and then wait 
			// some more.

			while(!GetOverlappedResult( m_idComDev, &m_osWrite, &dwBytesWritten, TRUE )) {

				dwError = GetLastError();
				if(dwError == ERROR_IO_INCOMPLETE)
					continue;		// normal result if not finished
				else {

					// an error occurred, try to recover
					ClearCommError( m_idComDev, &dwErrorFlags, &ComStat );
					break;
				}
			}

		}	else	{						 	

			m_rs232_out_state = RS232_ERROR;
			if (pView != NULL)
				pView->ShowRS232State();

			// some other error occurred
			ClearCommError( m_idComDev, &dwErrorFlags, &ComStat );
			_SOUND(m_sndHcsError);

			return FALSE;
		}

	}

	m_rs232_out_state = RS232_NOT_ACTIVE;
	if (pView != NULL)
		pView->ShowRS232State();

	return TRUE;
}



// High level I/O routines
void CWinHcsStatusDoc::SendMessageToHcs(char *pData, unsigned long sz)
{
	(void) WriteBlockToHcs(pData, sz);
}



// High level I/O routines
BOOL CWinHcsStatusDoc::SendMessageWithReplyToHcs(
			char *pData, unsigned long sz,
			char *pReply, unsigned long repSz)

{
	if (WriteBlockToHcs(pData, sz) == FALSE)
		return FALSE;

	m_replyP    = pReply;
	m_replySz   = repSz;

	return TRUE;
}


void CWinHcsStatusDoc::DownloadNewProgram(char *dataP, long size)
{
	m_prgSizeToSend	= size;
	m_prgDataP		= dataP;

	// Send command to start process
	SendMessageToHcs("!\x06", 2);
}


void CWinHcsStatusDoc::ReceiveAndProcessHcsMessages(CMainFrame *pFrame)
{
#define				INBUFFER_SZ		(64 * 1024)

	time_t				updateTime = NULL;
	time_t				tm;
	DWORD				dwEvtMask;
	OVERLAPPED			os;
	BOOL				fDone;
	BOOL				fUpdateView = FALSE;
	int					nLength, nPrevLength, nCount, nTempCnt, nOpCode, i;
	unsigned long		sz, idx;
	long				lVal, id;
	char				*pArg, *pC, prev_sec = 0;
	short				val[16], *pShort;
	char				buffer[INBUFFER_SZ];


	memset( &os, 0, sizeof(OVERLAPPED));
	memset( buffer, 0, INBUFFER_SZ);

	// create I/O event used for overlapped read
	os.hEvent = CreateEvent( NULL,		// no security
							TRUE,		// explicit reset req
							FALSE,		// initial event reset
							NULL );		// no name

	if (os.hEvent == NULL)	{

		AfxMessageBox(IDS_ERR_THREAD, MB_OK, 0);
		return;
	}

	if (!SetCommMask( m_idComDev, EV_RXCHAR ))
	  return;

	nLength		= 0;
	nPrevLength	= 0;

	while ( m_fConnected )	{

		dwEvtMask = 0;
		if (!WaitCommEvent( m_idComDev, &dwEvtMask, &os )) {

			if(GetLastError() == ERROR_IO_PENDING)	{
				GetOverlappedResult( m_idComDev, &os, &sz, TRUE );
			}

		}


		if ((dwEvtMask & EV_RXCHAR) == EV_RXCHAR)	{

			do	{

				memset(&buffer[nPrevLength], 0, INBUFFER_SZ - nPrevLength);

				if ((nLength = ReadBlockFromHcs( &buffer[nPrevLength], INBUFFER_SZ - nPrevLength)) > 0)	{

					// Add data we already buffered
					nLength += nPrevLength;

					// Dispatch message
					nCount = 0;
					fDone = FALSE;

					while (nCount < nLength && fDone == FALSE) {

						// Reply to command or Event?
						if (buffer[ nCount ] != '$') {

							// Special characters for program load
							if (buffer[ nCount ] == '^') {

								// Send the program
								if (m_prgSizeToSend > 0 && m_prgDataP != NULL) {

									idx = 0;

									while (idx < m_prgSizeToSend) {

										sz = m_prgSizeToSend - idx;
										if (sz > 100)		sz = 100;

										(void) WriteBlockToHcs(&m_prgDataP[idx], sz);
										idx += sz;

										pFrame->UpdateProgress(HcsLdPrgProgress, (long) idx);
									}

									free(m_prgDataP);

									m_prgSizeToSend = 0;
									m_prgDataP		= NULL;

								}	else

									pFrame->UpdateProgress(HcsLdPrgOK, 0);

							} else {

								if (buffer[ nCount ] == '#')
									pFrame->UpdateProgress(HcsLdPrgVer, 0);
								else
									if (buffer[ nCount ] == '@')
										pFrame->UpdateProgress(HcsLdPrgSize, 0);
							}

							nCount++;

						} else {

							// Ensure we at least have the opcode
							nTempCnt = nCount + 2;
							if (nTempCnt >= nLength) {

								fDone = TRUE;

							} else {

								nTempCnt = nCount + 1;

								nOpCode = (int)		buffer[ nTempCnt ] & 255;
								pArg    = (char *)	&buffer[nTempCnt + 1];

								switch ( nOpCode ) {
								default:
									lVal = nOpCode;
//									pFrame->UpdateProgress(HcsInfo, lVal);
									nCount += 2;	// skip $ and opCode
									break;

								case 0x08:			// Log size reply
									if ((nCount + 4) > nLength) {
										fDone = TRUE;
									} else {

										m_logNbEntries		= *((short *) pArg);
										m_logSizeReceived	= TRUE;

										pFrame->UpdateProgress(HcsGotLogSize, m_logNbEntries);

										nCount += 4;	// skip message
									}
									break;

								case 0x09:			// Log data reply
									i = nCount + 2;
									nTempCnt = 0;

									while (nTempCnt < 2 && fDone == FALSE) {

										if ((i + 2) >= nLength) {

											fDone = TRUE;

										} else {

											pC = (char *) &lVal;
											pC[0] = buffer[i];
											pC[1] = buffer[i + 1];
											pC[2] = buffer[i + 2];
											pC[3] = (char) 0xff;

											if (lVal == 0xffffffff)
											  nTempCnt = i + 3;
											else
											  i += 1;
										}
									}

									if (fDone == FALSE && nTempCnt > 0) {

										sz = nTempCnt - nCount - 2;

										memcpy(m_logData, pArg, sz);
										m_logDataSz = sz - 3;
										m_logPos	= 0;


										// Copy with 0xffffff terminator
										if (m_replyP != NULL && m_replySz >= sz)
											memcpy(m_replyP, pArg, sz);

										// but only report the number of entries
										lVal = sz - 3;
										pFrame->UpdateProgress(HcsGotLogData, lVal);

										nCount = nTempCnt;	// skip message

										m_logReceived = TRUE; // flag all received

									} else {

										lVal = nLength - nCount - 2;
										pFrame->UpdateProgress(HcsLogSizeProgress, lVal);
									}
									break;

								case 0x24:			// Variable reply
									if ((nCount + 5) > nLength) {
										fDone = TRUE;
									} else {

										pShort = (short *) (pArg + 1);

										id	 = (long) *pArg;
										lVal = (long) *pShort;

										m_XPRESS_Vars.SetIdValue(id, lVal);

//										val[0] = (short) *pArg;
//										val[1] = *pShort;
//										lVal = MAKELONG(val[1], val[0]);
//										pFrame->UpdateProgress(HcsGotVar, lVal);

										nCount += 5;	// skip message
									}
									break;

								// Events...
								case 0x80:
									if ((nCount + 10) > nLength) {
										fDone = TRUE;
									} else {
										for (i = 0 ; i < 8 ; i++)
											m_HcsTime[i] = _BCD_TO_INT( pArg[i] );
										nCount += 10;	// skip message
									}
									break;

								case 0x81:
									if ((nCount + 5) > nLength) {
										fDone = TRUE;
									} else {
										lVal = (long) *pArg;
										pArg++;
										m_X10.SetGroup(lVal, pArg);
										nCount += 5;	// skip message
									}
									break;

								case 0x82:
									if ((nCount + 4) > nLength) {
										fDone = TRUE;
									} else {
										lVal = (long) *pArg;
										pArg++;
										m_HcsInputs.SetGroup(lVal, pArg);
										nCount += 4;	// skip message
									}
									break;

								case 0x83:
									if ((nCount + 4) > nLength) {
										fDone = TRUE;
									} else {
										lVal = (long) *pArg;
										pArg++;
										m_HcsOutputs.SetGroup(lVal, pArg);
										nCount += 4;	// skip message
									}
									break;

								case 0x84:
									if ((nCount + 19) > nLength) {
										fDone = TRUE;
									} else {
										lVal = (long) *pArg;

										pArg++;
										m_ADCs.SetGroup(lVal, pArg);
										nCount += 19;	// skip message
									}
									break;

								case 0x85:
									if ((nCount + 7) > nLength) {
										fDone = TRUE;
									} else {
										memset(val, 0, 8 * sizeof(short));

										lVal = (long) *pArg;
										pShort = (short *) (pArg + 1);

										m_DACs.SetIdValue(lVal    , pShort[0]);
										m_DACs.SetIdValue(lVal + 1, pShort[1]);

										nCount += 7;	// skip message
									}
									break;

								case 0x86:
									if ((nCount + 4) > nLength) {
										fDone = TRUE;
									} else {
										i = (*pArg >> 4) & 0xf;
										lVal = (long) (*pArg & 0xf);
										pArg++;
										switch (i) {
										default:
											break;
										case 0:
											m_Modules_PL.SetGroup(lVal, pArg);
											break;
										case 1:
											m_Modules_MCIR.SetGroup(lVal, pArg);
											break;
										case 2:
											m_Modules_LCD.SetGroup(lVal, pArg);
											break;
										case 3:
											m_Modules_DIO.SetGroup(lVal, pArg);
											break;
										case 4:
											m_Modules_AMAN.SetGroup(lVal, pArg);
											break;
										case 5:	// DIOP modules
											break;
										}
										nCount += 4;	// skip message
									}
									break;

								case 0x87:
									if ((nCount + 4) > nLength) {
										fDone = TRUE;
									} else {
										lVal = (long) *pArg;
										pArg++;
										m_Netbits.SetGroup(lVal, pArg);
										nCount += 4;	// skip message
									}
									break;

								case 0x88:
									i = nCount + 2;
									while (i < nLength && buffer[i] != 0) {
										if (buffer[i] == '`')
											buffer[i] = '"';
										else if (buffer[i] == 0x0A || buffer[i] == 0x0D)
											buffer[i] = ' ';

										i++;
									}

									if (i < nLength) {

										if (pArg[0] == '#') {

											CString str( &pArg[1] );
											m_HcsCommands.AddTail(str);

										} else {

											CString str(pArg);

											m_ConsoleMessages.AddTail(str);
											fUpdateView = TRUE;
										}

										nCount = i + 1;	// skip message

									} else {
										fDone = TRUE;
									}
									break;
								}
							}
						}
					}
					// Copy left data to beginning
					i = nLength - nCount;
					if (i > 0) {

						memcpy(buffer, &buffer[nCount], i);
						nPrevLength = i;

					} else {

						nPrevLength = 0;
					}
				}
			} while ( nLength > 0 );

			// Post view refresh?
			tm = time(NULL);

			if (fUpdateView == TRUE || updateTime != tm) {

				updateTime = tm;

				if (m_rs232_in_state == RS232_ACTIVE)
					m_rs232_in_state = RS232_NOT_ACTIVE;
				else
					m_rs232_in_state = RS232_ACTIVE;

				pFrame->UpdateHcsView();

				fUpdateView = FALSE;
			}
		}	// EV_RXCHAR detection
	}	// While m_fConnected is TRUE

	// get rid of event handle
	CloseHandle( os.hEvent );

	// clear information in structure (kind of a "we're done flag")
	m_dwThreadID	= 0;
	m_hWatchThread	= NULL;

	return;
}


/////////////////////////////////////////////////////////////////////////////
// Get references

void  CWinHcsStatusDoc::SetNbBUFIO(long n)
{
	if (n >= 0 && n < 2)
		m_Module_BUFIO = n;
}

long  CWinHcsStatusDoc::GetNbBUFIO(void)
{
	return m_Module_BUFIO;
}

CHcsIo *  CWinHcsStatusDoc::GetHcsPLs(void)
{
	return &m_Modules_PL;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsMCIRs(void)
{
	return &m_Modules_MCIR;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsLCDs(void)
{
	return &m_Modules_LCD;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsDIOs(void)
{
	return &m_Modules_DIO;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsANSWERMANs(void)
{
	return &m_Modules_AMAN;
}

CHcsIo *  CWinHcsStatusDoc::GetHcsInputs(void)
{
	return &m_HcsInputs;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsOutputs(void)
{
	return &m_HcsOutputs;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsNetbits(void)
{
	return &m_Netbits;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsADCs(void)
{
	return &m_ADCs;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsDACs(void)
{
	return &m_DACs;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsX10(void)
{
	return &m_X10;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsVariables(void)
{
	return &m_XPRESS_Vars;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsTimers(void)
{
	return &m_XPRESS_Timers;
}
CHcsIo *  CWinHcsStatusDoc::GetHcsLogIDs(void)
{
	return &m_XPRESS_Logs;
}


void CWinHcsStatusDoc::SetHcsTime(void)
{
	CTime			m( time(NULL) );
	int				t;
	char			cmd[16];
	
	cmd[0] = '!';
	cmd[1] = 0x05;
	cmd[2] = 0;

	t = m.GetSecond();
	cmd[3] = _INT_TO_BCD( t );

	t = m.GetMinute();
	cmd[4] = _INT_TO_BCD( t );

	t = m.GetHour();
	cmd[5] = _INT_TO_BCD( t );

	t = m.GetDayOfWeek();
	cmd[6] = _INT_TO_BCD( t );

	t = m.GetDay();
	cmd[7] = _INT_TO_BCD( t );

	t = m.GetMonth();
	cmd[8] = _INT_TO_BCD( t );

	t = m.GetYear() - 1900;
	cmd[9] = _INT_TO_BCD( t );
	
	SendMessageToHcs(cmd, 10);
	SendMessageToHcs("!\x02", 2);
}



char * CWinHcsStatusDoc::GetHcsTime(void)
{
	long			i, y;
	char			*dP, *mP;
	static char		*dayTbl[] =
			{ "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat" };
	static char		*monthTbl[] = 
			{ "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			"Jul", "Aug", "Sep", "Oct", "Nov", "Dev" };
	static char		b[128];

	if (IsConnectedToHcs() == FALSE)
		return _T("");

	i  =  m_HcsTime[4] - 1;
	if (i < 0 || i > 6)		dP = "???";
	else					dP = dayTbl[ i ];

	i  =  m_HcsTime[6] - 1; 
	if (i < 0 || i > 11)	mP = "???";
	else					mP = monthTbl[ i ];

	if ((long) m_HcsTime[7] > 70)
		y = (long)m_HcsTime[7] + 1900;
	else
		y = (long)m_HcsTime[7] + 2000;

	sprintf(b, "%s, %s %d, %d %02d:%02d:%02d ",
				dP,	mP,	(long)m_HcsTime[5], y,
				(long)m_HcsTime[3], (long)m_HcsTime[2], (long)m_HcsTime[1]);
				
	return b;
}


BOOL CWinHcsStatusDoc::GetHcsTime(CTime &tm)
{
	long			y;

	if (IsConnectedToHcs() == FALSE)
		return FALSE;

	if ((long) m_HcsTime[7] > 70)	y = (long)m_HcsTime[7] + 1900;
	else							y = (long)m_HcsTime[7] + 2000;

	CTime	hcsTime(y, m_HcsTime[6], (long)m_HcsTime[5],(long)m_HcsTime[3], (long)m_HcsTime[2], (long)m_HcsTime[1], -1);
	tm = hcsTime;

	return TRUE;
}



char * CWinHcsStatusDoc::GetLastConsoleMsg(void)
{
	static char msg[1024], *sP;

	if (m_ConsoleMessages.IsEmpty() == TRUE)
		return NULL;

	CString str = m_ConsoleMessages.RemoveHead();

	sP = str.GetBuffer(1024);
	strncpy(msg, sP, 1024);
	msg[1023] = 0;

	return msg;
}


//////////////////////////////////////////////////////////
void CWinHcsStatusDoc::GetRS232Settings(CString &comPort, BOOL  *pUsingModem, CString & modemInitCmd)
{
	if (m_bPort == 2)		comPort = _T("COM2");
	else					comPort = _T("COM1");

	*pUsingModem	= m_UsingModem;
	modemInitCmd	= m_ModemInitCommand;
}

void CWinHcsStatusDoc::SetRS232Settings(CString &comPort, BOOL bUsingModem, CString & modemInitCmd)
{
	CWinHcsApp		*pApp	= (CWinHcsApp *) AfxGetApp();
	CString			s1( _T("COM2") );
	char			szPort[16];


	if (comPort == s1)		m_bPort = 2;
	else					m_bPort = 1;

	m_UsingModem		= bUsingModem;
	m_ModemInitCommand	= modemInitCmd;


	// Save to Registry
	sprintf(szPort, "COM%d", m_bPort );

	pApp->SetProfileCString(	IDS_REG_SETTINGS,	IDS_REG_COM,		szPort);
	pApp->SetProfileCString(	IDS_REG_SETTINGS,	IDS_REG_MODEM,		m_UsingModem);
	pApp->SetProfileCString(	IDS_REG_SETTINGS,	IDS_REG_MODEM_INIT,	m_ModemInitCommand.GetBuffer(256));
}

////////////////////////////////////////////////////////
void CWinHcsStatusDoc::SetTimeUpdateRate(long updateRate)
{
	CWinHcsApp		*pApp	= (CWinHcsApp *) AfxGetApp();
	CString			key;
	char			arg;
	char			cmd[8];

	switch (updateRate) {
	case TIME_UPDATE_NONE:
		m_HcsUpdatePeriod = updateRate;
		arg = 0;
		break;
	case TIME_UPDATE_SLOW:
		m_HcsUpdatePeriod = updateRate;
		arg = 2;
		break;
	default:
	case TIME_UPDATE_FAST:
		m_HcsUpdatePeriod = TIME_UPDATE_FAST;
		arg = 1;
		break;
	}

	// Save to registry
	pApp->SetProfileLong(	IDS_REG_HCS, IDS_REG_TIME_UPD_RATE, m_HcsUpdatePeriod);


	// And now update HCS if connected
	if (IsConnectedToHcs()) {

		cmd[0] = '!';
		cmd[1] = 0x03;
		cmd[2] = arg;

		SendMessageToHcs(cmd, 3);
	}
}

long CWinHcsStatusDoc::GetTimeUpdateRate()
{
	return m_HcsUpdatePeriod;
}




void CWinHcsStatusDoc::SetLogUpdateSettings(long sizeUpdateRate, BOOL logUpdateAuto, BOOL clearAfterStore)
{
	CWinHcsApp		*pApp	= (CWinHcsApp *) AfxGetApp();

	if (sizeUpdateRate < 0 || sizeUpdateRate > 60)
		sizeUpdateRate = 1;

	m_logUpdateRate  = sizeUpdateRate;
	m_logAutoClear   = clearAfterStore;
	m_logUpdateAuto	 = logUpdateAuto;

	// Save to registry
	pApp->SetProfileLong(IDS_REG_HCS,	IDS_REG_LOG_UPD_RATE,	m_logUpdateRate);
	pApp->SetProfileLong(IDS_REG_HCS,	IDS_REG_LOG_UPD_AUTO,	m_logUpdateAuto);
	pApp->SetProfileLong(IDS_REG_HCS,	IDS_REG_LOG_CLEAR,		m_logAutoClear);
}


void CWinHcsStatusDoc::GetLogUpdateSettings(long *sizeUpdateRate, BOOL *logUpdateAuto, BOOL *clearAfterStore)
{
	*sizeUpdateRate  = m_logUpdateRate;
	*logUpdateAuto	 = m_logUpdateAuto;
	*clearAfterStore = m_logAutoClear;
}


void CWinHcsStatusDoc::ClearLogEntries(void)
{
	SendMessageToHcs("!\x07", 2);
	SendMessageToHcs("!\x08", 2);
}


void CWinHcsStatusDoc::QueryLogSize(void)
{
	((CMainFrame*) AfxGetMainWnd())->ClearLogEntries();

	m_logSizeReceived = FALSE;
	SendMessageToHcs("!\x08", 2);
}


void CWinHcsStatusDoc::AddLogEntry(short id, short value)
{
	char	cmd[8];

	cmd[0] = '!';
	cmd[1] = 0x0a;
	cmd[2] = (char) id;
	*((short *) &cmd[3]) = (short) value;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x08", 2);
}


void CWinHcsStatusDoc::UploadLogData(void)
{
	m_logReceived = FALSE;
	SendMessageToHcs("!\x09", 2);
}


BOOL CWinHcsStatusDoc::GetFirstLogEntry(CTime &tm, long *idP, long *valueP)
{
	m_logPos  = 0;

	return GetNextLogEntry(tm, idP, valueP);
}

BOOL CWinHcsStatusDoc::GetNextLogEntry(CTime &tm, long *idP, long *valueP)
{
	unsigned char *sP = (unsigned char *) m_logData;
	short			y, o, d, h, m, s;

	if (m_logPos >= m_logDataSz)
		return FALSE;

	if ((long) m_HcsTime[7] > 70)	y = (long)m_HcsTime[7] + 1900;
	else							y = (long)m_HcsTime[7] + 2000;

	o = sP[m_logPos+3];
	d = sP[m_logPos+4];
	h = sP[m_logPos+5];
	m = sP[m_logPos+6];
	s = sP[m_logPos+7];

	CTime t(y, o, d, h, m, s, -1);

	tm = t;
	*idP	= sP[m_logPos];
	*valueP	= (long)(sP[m_logPos+1] + (sP[m_logPos+2] * 256));

	m_logPos += 8;

	return TRUE;
}


long CWinHcsStatusDoc::GetUploadedLogData(char **pLogData)
{
	m_logPos  = 0;
	*pLogData = m_logData;

	return m_logDataSz;
}




/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
//
//	Automation support
//
/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////


// --- General

VARIANT CWinHcsStatusDoc::AutomationGetApplication() 
{
	CWinHcsApp		*pApp    = (CWinHcsApp *) AfxGetApp();
	VARIANT			vaResult;

	VariantInit(&vaResult);

	vaResult.vt = VT_DISPATCH;
	vaResult.punkVal = pApp->GetIDispatch(TRUE);

	return vaResult;
}


// --- Connection


BOOL CWinHcsStatusDoc::AutomationGetIsConnected() 
{
	return IsConnectedToHcs();
}

void CWinHcsStatusDoc::AutomationSetIsConnected(BOOL bNewValue) 
{
	if (bNewValue == TRUE) {

		if (IsConnectedToHcs() == FALSE)
			OpenConnectionToHcs();

	} else {

		if (IsConnectedToHcs() == TRUE)
			CloseConnectionToHcs();
	}
}


// --- HCS Time

void CWinHcsStatusDoc::AutomationSetTime() 
{
	SetHcsTime();
}


DATE CWinHcsStatusDoc::AutomationGetTime() 
{
	CTime	hcsTime;
	BOOL	bVal;

	bVal = GetHcsTime(hcsTime);

	if (bVal == FALSE)
		return (DATE) 0;

	COleDateTime	t(hcsTime.GetTime());
	DATE			dateTime = t;

	return dateTime;
}


// --- Console Message


BOOL CWinHcsStatusDoc::AutomationGetNewConsoleMsg() 
{
	BOOL	ret = FALSE;

	if (m_ConsoleMessages.IsEmpty() == FALSE)
		ret = TRUE;

	return ret;
}


BSTR CWinHcsStatusDoc::AutomationGetLastConsoleMsg() 
{
	CString strResult;

	if (m_ConsoleMessages.IsEmpty() == FALSE)
		strResult = m_ConsoleMessages.RemoveHead();

	return strResult.AllocSysString();
}



// --- Log Data

void CWinHcsStatusDoc::AutomationClearLogEntries() 
{
	ClearLogEntries();
}


long CWinHcsStatusDoc::AutomationGetNbLogEntries() 
{
	long			n = 100;

	if (IsConnectedToHcs() == FALSE)
		return 0;

	
	QueryLogSize();

	while (m_logSizeReceived == FALSE && n > 0) {
		n--;
		::Sleep(100);
	}

	return m_logNbEntries;
}


void CWinHcsStatusDoc::AutomationAddLogEntry(short id, short value) 
{
	AddLogEntry(id, value);
}


BOOL CWinHcsStatusDoc::AutomationGetLogData() 
{
	long			n = 100;

	if (IsConnectedToHcs() == FALSE)
		return FALSE;

	UploadLogData();

	while (m_logReceived == FALSE && n > 0) {
		n--;
		::Sleep(100);
	}

	return m_logReceived;
}

BOOL CWinHcsStatusDoc::AutomationGetFirstLogEntry() 
{
	BOOL	ret;
	CTime	tm;

	if ((ret = GetFirstLogEntry(tm, &m_hcsLogID, &m_hcsLogValue)) == TRUE) {

		m_hcsLogYear	= tm.GetYear();
		m_hcsLogMonth	= tm.GetMonth();
		m_hcsLogDay		= tm.GetDay();
		m_hcsLogHour	= tm.GetHour();
		m_hcsLogMinute	= tm.GetMinute();
		m_hcsLogSecond	= tm.GetSecond();
	}

	return ret;
}

BOOL CWinHcsStatusDoc::AutomationGetNextLogEntry() 
{
	BOOL	ret;
	CTime	tm;

	if ((ret = GetNextLogEntry(tm, &m_hcsLogID, &m_hcsLogValue)) == TRUE) {

		m_hcsLogYear	= tm.GetYear();
		m_hcsLogMonth	= tm.GetMonth();
		m_hcsLogDay		= tm.GetDay();
		m_hcsLogHour	= tm.GetHour();
		m_hcsLogMinute	= tm.GetMinute();
		m_hcsLogSecond	= tm.GetSecond();
	}

	return ret;
}


BOOL CWinHcsStatusDoc::AutomationGetFirstDBLogEntry(long IDFrom, long IDTo, LPCTSTR DateFrom, LPCTSTR DateTo) 
{
	CString			str1, str2;

	if (m_dbRec.IsOpen()) {

		try {
			m_dbRec.Close();
		}
		catch(CDBException *pEx)
		{
			delete pEx;
		}
	}

	if (m_db.IsOpen()) {

		try {
			m_db.Close();
		}
		catch(CDBException *pEx)
		{
			delete pEx;
		}
	}

	// Setup Filter mechanism
	if (IDFrom >= 0) {

		if (IDTo >= 0)
			str1.Format(_T("EventID >= %d AND EventID <= %d"), IDFrom, IDTo);
		else
			str1.Format(_T("EventID >= %d"), IDFrom);

	} else if (IDTo >= 0) {

		str1.Format(_T("EventID <= %d"), IDTo);
	}


	if (DateFrom == NULL || *DateFrom == 0) {

		if (DateTo != NULL && *DateTo != 0)
			str2.Format(_T("EventDateTime <= #%s#"), DateTo);

	} else {

		if (DateTo == NULL || *DateTo == 0)
			str2.Format(_T("EventDateTime >= #%s#"), DateFrom);
		else
			str2.Format(_T("EventDateTime >= #%s# AND EventDateTime <= #%s#"), DateFrom, DateTo);
	}


	if (str1.IsEmpty() == TRUE) {

		if (str2.IsEmpty() == FALSE)
			m_dbRec.m_strFilter = str2;

	} else {

		if (str2.IsEmpty() == TRUE)
			m_dbRec.m_strFilter = str1;
		else
			m_dbRec.m_strFilter.Format(_T("%s AND %s"), str1, str2);
	}


	// Try opening the database
	try {
		if (m_db.IsOpen() == FALSE)
			m_db.Open(m_OdbcDSN.GetBuffer(80), FALSE, FALSE, _T("ODBC;"), TRUE);
	}
	catch(CDBException *pEx)
	{
		CString str( _T("Database LogData : Could not open database.") );
		str += pEx->m_strError;

		((CMainFrame*) AfxGetMainWnd())->DlgBarAddMessage(str.GetBuffer(128));
		delete pEx;

		return FALSE;
	}

	// Open the recordset and fetch the items
	try {
		if (m_dbRec.IsOpen() == FALSE)
			m_dbRec.Open(AFX_DB_USE_DEFAULT_TYPE, m_OdbcTable.GetBuffer(80), 0);
	}
	catch(CDBException *pEx)
	{
		CString str( _T("Database LogData: Could not open database table.") );
		str += pEx->m_strError;

		((CMainFrame*) AfxGetMainWnd())->DlgBarAddMessage(str.GetBuffer(128));

		delete pEx;
		m_db.Close();

		return FALSE;
	}

	return AutomationGetDBNextLogEntry();
}

BOOL CWinHcsStatusDoc::AutomationGetDBNextLogEntry() 
{
	if ( m_dbRec.IsEOF() ) {

		m_dbRec.Close();
		m_db.Close();

		return FALSE;
	}

	m_hcsLogYear	= m_dbRec.m_EventDateTime.year;
	m_hcsLogMonth	= m_dbRec.m_EventDateTime.month;
	m_hcsLogDay		= m_dbRec.m_EventDateTime.day;
	m_hcsLogHour	= m_dbRec.m_EventDateTime.hour;
	m_hcsLogMinute	= m_dbRec.m_EventDateTime.minute;
	m_hcsLogSecond	= m_dbRec.m_EventDateTime.second;
	m_hcsLogID		= m_dbRec.m_EventID;
	m_hcsLogValue	= m_dbRec.m_EventValue;

	// Get next record
	m_dbRec.MoveNext();

	return TRUE;
}


// --- Send Voice & Network messages

void CWinHcsStatusDoc::AutomationSpeak(LPCTSTR TextToSpeak) 
{
	CString str(TextToSpeak);
	char	*cmdP;
	long	sz;

	if (IsConnectedToHcs() == FALSE)
		return;

	sz = str.GetLength() + 3;

	if (sz > 3) {

		cmdP = (char *) malloc(sz);
		ASSERT(cmdP != NULL);

		cmdP[0] = '!';
		cmdP[1] = 0x31;
		strcpy(&cmdP[2], str.GetBuffer(sz));

		SendMessageToHcs(cmdP, sz);
		free(cmdP);
	}
}


void CWinHcsStatusDoc::AutomationSendToNetwork(LPCTSTR cmd) 
{
	CString str(cmd);
	char	*cmdP;
	long	sz;

	if (IsConnectedToHcs() == FALSE)
		return;

	sz = str.GetLength() + 3;

	if (sz > 3) {

		cmdP = (char *) malloc(sz);
		ASSERT(cmdP != NULL);

		cmdP[0] = '!';
		cmdP[1] = 0x30;
		strcpy(&cmdP[2], str.GetBuffer(sz));

		SendMessageToHcs(cmdP, sz);
		free(cmdP);
	}
}


// --- Set I/O Values (ADC, DAC, Output, Input, Netbit, Variable, X10)

void CWinHcsStatusDoc::AutomationSetDac(short id, short value) 
{
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsDACs()->SetBeingChanged(id);

	cmd[0] = '!';
	cmd[1] = 0x1e;
	cmd[2] = (char) id;
	*((unsigned short *) &cmd[3]) = (unsigned short) value;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetADC(short id, BOOL transparent, short value) 
{
	unsigned short	v = (unsigned short) value;
	char			cmd[8];

	if (transparent == TRUE)
		v = 4096;

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsADCs()->SetBeingChanged(id);

	cmd[0] = '!';
	cmd[1] = 0x1b;
	cmd[2] = (char) id;
	*((unsigned short *) &cmd[3]) = v;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}


void CWinHcsStatusDoc::AutomationSetOutput(short id, BOOL value) 
{
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsOutputs()->SetBeingChanged(id);

	cmd[0] = '!';
	cmd[1] = 0x18;
	cmd[2] = (char) id;
	cmd[3] = (char) value;

	SendMessageToHcs(cmd, 4);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetInput(short id, BOOL transparent, BOOL value) 
{
	char	v = (char) value;
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	if (transparent == TRUE)
		v = 2;

	GetHcsInputs()->SetBeingChanged(id);

	cmd[0] = '!';
	cmd[1] = 0x15;
	cmd[2] = (char) id;
	cmd[3] = (char) v;

	SendMessageToHcs(cmd, 4);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetNetbit(short id, BOOL value) 
{
	unsigned short	v = (unsigned short) id;
	char			cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsNetbits()->SetBeingChanged(id);

	cmd[0] = '!';
	cmd[1] = 0x23;
	*((unsigned short *) &cmd[2]) = v;
	cmd[4] = (char) value;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetVariable(short id, short value) 
{
	char			cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsVariables()->SetBeingChanged(id);

	cmd[0] = '!';
	cmd[1] = 0x25;
	cmd[2] = (char) id;
	*((short *) &cmd[3]) = value;

	SendMessageToHcs(cmd, 5);


	cmd[0] = '!';
	cmd[1] = 0x24;
	cmd[2] = (char) id;

	SendMessageToHcs(cmd, 3);
}


void CWinHcsStatusDoc::AutomationSetX10OnOrOff(short HouseCodeAndModule, BOOL OnState) 
{
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsX10()->SetBeingChanged(HouseCodeAndModule);

	cmd[0] = '!';
	cmd[1] = 0x12;
	cmd[2] = (char) HouseCodeAndModule;
	if (OnState == TRUE) 
		cmd[3] = (char) 2;
	else
		cmd[3] = (char) 3;

	cmd[4] = (char) 0;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetX10DimOrBright(short HouseCodeAndModule, short DimBrightSteps) 
{
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	GetHcsX10()->SetBeingChanged(HouseCodeAndModule);

	cmd[0] = '!';
	cmd[1] = 0x12;
	cmd[2] = (char) HouseCodeAndModule;
	if (DimBrightSteps >= 0) 
		cmd[3] = (char) 5;
	else
		cmd[3] = (char) 4;

	cmd[4] = (char) abs(DimBrightSteps);

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetX10AllOff(short HouseCode) 
{
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	cmd[0] = '!';
	cmd[1] = 0x12;
	cmd[2] = (char) (HouseCode & 0x0f << 4);
	cmd[3] = (char) 0;
	cmd[4] = (char) 0;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}

void CWinHcsStatusDoc::AutomationSetX10AllOn(short HouseCode) 
{
	char	cmd[8];

	if (IsConnectedToHcs() == FALSE)
		return;

	cmd[0] = '!';
	cmd[1] = 0x12;
	cmd[2] = (char) (HouseCode & 0x0f << 4);
	cmd[3] = (char) 1;
	cmd[4] = (char) 0;

	SendMessageToHcs(cmd, 5);
	SendMessageToHcs("!\x02", 2);
}


// --- Get I/O Values (ADC, DAC, Output, Input, Netbit, Variable, X10)

short CWinHcsStatusDoc::AutomationGetADC(short id) 
{
	CHcsIo	*pIO = GetHcsADCs();
	return (short) pIO->GetIdValue((long) id);
}

short CWinHcsStatusDoc::AutomationGetDAC(short id) 
{
	CHcsIo	*pIO = GetHcsDACs();
	return (short) pIO->GetIdValue((long) id);
}

BOOL CWinHcsStatusDoc::AutomationGetOutput(short id) 
{
	CHcsIo	*pIO = GetHcsOutputs();
	return (BOOL) pIO->GetIdValue((long) id);
}

BOOL CWinHcsStatusDoc::AutomationGetInput(short id) 
{
	CHcsIo	*pIO = GetHcsInputs();
	return (BOOL) pIO->GetIdValue((long) id);
}

BOOL CWinHcsStatusDoc::AutomationGetNetbit(short id) 
{
	CHcsIo	*pIO = GetHcsNetbits();
	return (BOOL) pIO->GetIdValue((long) id);
}

short CWinHcsStatusDoc::AutomationGetVariable(short id) 
{
	CHcsIo	*pIO = GetHcsVariables();
	return (short) pIO->GetIdValue((long) id);
}

BOOL CWinHcsStatusDoc::AutomationGetX10(short HouseCodeAndModule) 
{
	CHcsIo	*pIO = GetHcsX10();
	return (BOOL) pIO->GetIdValue((long) HouseCodeAndModule);
}

// --- Set I/O Properties (ADC, DAC, Output, Input, Netbit, Variable, X10)

void CWinHcsStatusDoc::AutomationSetADCProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsADCs();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetDACProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsDACs();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetOutputProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsOutputs();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetInputProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsInputs();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetNetbitProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsNetbits();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetVariableProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsVariables();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetX10Properties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsX10();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetLogIDProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsLogIDs();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}

void CWinHcsStatusDoc::AutomationSetTimerProperties(short id, BOOL used, LPCTSTR label) 
{
	CHcsIo	*pIO = GetHcsTimers();

	if (used == FALSE) 	pIO->SetNotInUse((long) id);
	else				pIO->SetInUse((long) id, (char *) label);
}


// --- Get I/O Properties (ADC, DAC, Output, Input, Netbit, Variable, X10, logIDs, Timers)



BOOL CWinHcsStatusDoc::AutomationIsADCInUse(short id) 
{
	return GetHcsADCs()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsDACInUse(short id) 
{
	return GetHcsDACs()->IsInUse((long) id);
}


BOOL CWinHcsStatusDoc::AutomationIsOutputInUse(short id) 
{
	return GetHcsOutputs()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsInputInUse(short id) 
{
	return GetHcsInputs()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsNetBitInUse(short id) 
{
	return GetHcsNetbits()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsVariableInUse(short id) 
{
	return GetHcsVariables()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsX10InUse(short id) 
{
	return GetHcsX10()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsLogIdInUse(short id) 
{
	return GetHcsLogIDs()->IsInUse((long) id);
}

BOOL CWinHcsStatusDoc::AutomationIsTimerInUse(short id) 
{
	return GetHcsTimers()->IsInUse((long) id);
}



BSTR CWinHcsStatusDoc::AutomationGetLabel(short id, CHcsIo *pIO) 
{
	char	*sP;
	CString strResult;
	
	if ((sP = pIO->GetIoLabel((long) id)) == NULL)
		sP = _T("");

	strResult = sP;

	return strResult.AllocSysString();
}

BSTR CWinHcsStatusDoc::AutomationGetADCLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsADCs());
}

BSTR CWinHcsStatusDoc::AutomationGetDACLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsDACs());
}

BSTR CWinHcsStatusDoc::AutomationGetOutputLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsOutputs());
}

BSTR CWinHcsStatusDoc::AutomationGetInputLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsInputs());
}

BSTR CWinHcsStatusDoc::AutomationGetNetbitLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsNetbits());
}

BSTR CWinHcsStatusDoc::AutomationGetVariableLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsVariables());
}

BSTR CWinHcsStatusDoc::AutomationGetX10Label(short id) 
{
	return AutomationGetLabel(id, GetHcsX10());
}

BSTR CWinHcsStatusDoc::AutomationGetLogIDLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsLogIDs());
}

BSTR CWinHcsStatusDoc::AutomationGetTimerLabel(short id) 
{
	return AutomationGetLabel(id, GetHcsTimers());
}



// --- Views

void CWinHcsStatusDoc::AutomationAddWindow() 
{
	CWinHcsApp	*pApp = (CWinHcsApp *) AfxGetApp();
	pApp->OnViewHcsstatus();
}


VARIANT CWinHcsStatusDoc::AutomationGetWindow() 
{
	POSITION			pos;
	CWinHcsStatusView	*pView;
	VARIANT				vaResult;

	VariantInit(&vaResult);
	vaResult.vt = VT_DISPATCH;
	vaResult.punkVal = NULL;

	// Does a view already exist ?
	if ((pos = GetFirstViewPosition()) == NULL)
		AutomationAddWindow();

	if ((pos = GetFirstViewPosition()) == NULL)
		return vaResult;


	pView = (CWinHcsStatusView *) GetNextView(pos);
	vaResult.punkVal = pView->GetIDispatch(TRUE);

	return vaResult;
}
